<?php
/**
 * SEO Dashboard Template
 * v3.2.0 - Main SEO health dashboard with scores, cannibalization detection, and recommendations
 * 
 * @package WebStudio SEO Pro
 */

if (!defined('ABSPATH')) exit;

// Calculate SEO Health Score
function wseo_calculate_health_score() {
    $score = 0;
    $max_score = 100;
    $issues = array();
    $warnings = array();
    $passed = array();
    
    // 1. Check if site title is set (5 points)
    $site_name = get_option('wseo_global_site_name', get_bloginfo('name'));
    if (!empty($site_name) && strlen($site_name) > 3) {
        $score += 5;
        $passed[] = 'Názov stránky je nastavený';
    } else {
        $issues[] = array('text' => 'Nastavte názov stránky', 'link' => admin_url('admin.php?page=webstudio-seo-global'));
    }
    
    // 2. Check home meta description (10 points)
    $home_desc = get_option('wseo_global_home_description', '');
    if (!empty($home_desc) && strlen($home_desc) >= 50) {
        $score += 10;
        $passed[] = 'Meta popis homepage je nastavený';
    } else {
        $issues[] = array('text' => 'Nastavte meta popis pre homepage (min. 50 znakov)', 'link' => admin_url('admin.php?page=webstudio-seo-global'));
    }
    
    // 3. Check Schema.org settings (15 points)
    $schema_type = get_option('wseo_schema_type', '');
    $schema_name = get_option('wseo_schema_name', '');
    if (!empty($schema_type) && !empty($schema_name)) {
        $score += 15;
        $passed[] = 'Schema.org je nakonfigurované';
    } else {
        $issues[] = array('text' => 'Nakonfigurujte Schema.org typ organizácie', 'link' => admin_url('admin.php?page=webstudio-seo-global#schema'));
    }
    
    // 4. Check if logo is set (5 points)
    $logo = get_option('wseo_schema_logo', '');
    if (!empty($logo)) {
        $score += 5;
        $passed[] = 'Logo je nastavené';
    } else {
        $warnings[] = array('text' => 'Pridajte logo organizácie', 'link' => admin_url('admin.php?page=webstudio-seo-global'));
    }
    
    // 5. Check social profiles (5 points) - v3.2: Check ALL social networks with CORRECT option names
    $social_fb = get_option('wseo_schema_facebook', '');
    $social_ig = get_option('wseo_schema_instagram', '');
    $social_tw = get_option('wseo_schema_twitter', '');
    $social_li = get_option('wseo_schema_linkedin', '');
    $social_yt = get_option('wseo_schema_youtube', '');
    $social_tt = get_option('wseo_schema_tiktok', '');
    $social_pi = get_option('wseo_schema_pinterest', '');
    
    // Check if ANY social network is configured
    $has_social = !empty($social_fb) || !empty($social_ig) || !empty($social_tw) || 
                  !empty($social_li) || !empty($social_yt) || !empty($social_tt) || !empty($social_pi);
    
    if ($has_social) {
        $score += 5;
        $passed[] = 'Sociálne siete sú prepojené';
    } else {
        $warnings[] = array('text' => 'Prepojte sociálne siete', 'link' => admin_url('admin.php?page=webstudio-seo-global#social'));
    }
    
    // 6. Check robots.txt (5 points) - v3.2: WSEO always generates robots.txt automatically
    // So this is ALWAYS configured - just give the points
    $score += 5;
    $passed[] = 'Robots.txt je aktívny (automaticky generovaný)';
    
    // 7. Check for broken links (10 points)
    global $wpdb;
    $broken_links = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wseo_link_map WHERE status = 'broken'");
    if ($broken_links === null || intval($broken_links) === 0) {
        $score += 10;
        $passed[] = 'Žiadne nefunkčné odkazy';
    } else {
        $issues[] = array('text' => 'Máte ' . intval($broken_links) . ' nefunkčných odkazov', 'link' => admin_url('admin.php?page=webstudio-seo-links'));
    }
    
    // 8. Check 404 errors (10 points)
    $errors_404 = get_option('wseo_404_log', array());
    $recent_404 = array_filter($errors_404, function($e) {
        return isset($e['time']) && $e['time'] > strtotime('-7 days');
    });
    if (count($recent_404) < 5) {
        $score += 10;
        $passed[] = 'Minimum 404 chýb za posledný týždeň';
    } else {
        $warnings[] = array('text' => count($recent_404) . ' 404 chýb za posledný týždeň', 'link' => admin_url('admin.php?page=webstudio-seo-redirects'));
    }
    
    // 9. Check images without alt (10 points) — v3.4: static cache to avoid duplicate query
    static $cached_images_no_alt = null;
    if ($cached_images_no_alt === null) {
        $cached_images_no_alt = $wpdb->get_var("
            SELECT COUNT(*) FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
            WHERE p.post_type = 'attachment' 
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value = '')
            LIMIT 1000
        ");
    }
    $images_no_alt = $cached_images_no_alt;
    if (intval($images_no_alt) < 10) {
        $score += 10;
        $passed[] = 'Väčšina obrázkov má alt text';
    } else {
        $issues[] = array('text' => intval($images_no_alt) . ' obrázkov nemá alt text', 'link' => admin_url('admin.php?page=webstudio-seo-images'));
    }
    
    // 10. Check SSL (5 points)
    if (is_ssl()) {
        $score += 5;
        $passed[] = 'SSL certifikát je aktívny';
    } else {
        $issues[] = array('text' => 'Stránka nepoužíva HTTPS', 'link' => '#');
    }
    
    // 11. Check XML Sitemap (5 points)
    $sitemap_enabled = get_option('wseo_sitemap_enabled', '1');
    if ($sitemap_enabled === '1') {
        $score += 5;
        $passed[] = 'XML Sitemap je aktívna';
    } else {
        $issues[] = array('text' => 'Aktivujte XML Sitemap', 'link' => admin_url('admin.php?page=webstudio-seo-files'));
    }
    
    // 12. Check llms.txt for AI (5 points)
    $llms_enabled = get_option('wseo_llms_enabled', '1');
    if ($llms_enabled === '1') {
        $score += 5;
        $passed[] = 'llms.txt pre AI je aktívny';
    } else {
        $warnings[] = array('text' => 'Aktivujte llms.txt pre AI vyhľadávače', 'link' => admin_url('admin.php?page=webstudio-seo-files'));
    }
    
    return array(
        'score' => min($score, $max_score),
        'max_score' => $max_score,
        'issues' => $issues,
        'warnings' => $warnings,
        'passed' => $passed
    );
}

// Keyword Cannibalization Detection
function wseo_detect_keyword_cannibalization() {
    global $wpdb;
    
    // Get all published posts/pages with their focus keywords and titles
    $posts = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type, p.post_name,
               pm_focus.meta_value as focus_keyword,
               pm_title.meta_value as seo_title
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm_focus ON p.ID = pm_focus.post_id AND pm_focus.meta_key = '_wseo_focus_keyword'
        LEFT JOIN {$wpdb->postmeta} pm_title ON p.ID = pm_title.post_id AND pm_title.meta_key = '_wseo_title'
        WHERE p.post_status = 'publish'
        AND p.post_type IN ('post', 'page', 'product')
        ORDER BY p.post_date DESC
        LIMIT 500
    ");
    
    $cannibalization = array();
    $keyword_map = array();
    
    foreach ($posts as $post) {
        // Get keywords to check (focus keyword + words from title)
        $keywords_to_check = array();
        
        // Focus keyword
        if (!empty($post->focus_keyword)) {
            $focus = mb_strtolower(trim($post->focus_keyword));
            if (strlen($focus) >= 3) {
                $keywords_to_check[] = $focus;
            }
        }
        
        // Extract significant words from title (3+ chars, not stop words)
        $stop_words = array('a', 'an', 'the', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for', 'of', 'with', 'by', 'from', 'as', 'is', 'was', 'are', 'were', 'been', 'be', 'have', 'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should', 'may', 'might', 'must', 'shall', 'can', 'need', 'dare', 'ought', 'used', 'ako', 'pre', 'pri', 'pod', 'nad', 'bez', 'cez', 'než', 'ani', 'ale', 'alebo', 'keď', 'kde', 'kto', 'čo', 'ten', 'tá', 'to', 'je', 'sú', 'bol', 'bola', 'boli', 'bude', 'budú');
        
        $title_words = preg_split('/[\s\-–—:,\.!\?]+/', mb_strtolower($post->post_title));
        foreach ($title_words as $word) {
            $word = trim($word);
            if (strlen($word) >= 4 && !in_array($word, $stop_words) && !is_numeric($word)) {
                $keywords_to_check[] = $word;
            }
        }
        
        // Check for duplicates
        foreach ($keywords_to_check as $keyword) {
            if (!isset($keyword_map[$keyword])) {
                $keyword_map[$keyword] = array();
            }
            $keyword_map[$keyword][] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'type' => $post->post_type,
                'slug' => $post->post_name,
                'is_focus' => ($keyword === mb_strtolower(trim($post->focus_keyword)))
            );
        }
    }
    
    // Filter to only keywords with multiple posts
    foreach ($keyword_map as $keyword => $posts) {
        if (count($posts) >= 2) {
            // Prioritize if focus keyword matches
            $has_focus = false;
            foreach ($posts as $p) {
                if ($p['is_focus']) $has_focus = true;
            }
            
            $cannibalization[] = array(
                'keyword' => $keyword,
                'posts' => $posts,
                'count' => count($posts),
                'severity' => $has_focus ? 'high' : 'medium'
            );
        }
    }
    
    // Sort by count (most conflicts first) then by severity
    usort($cannibalization, function($a, $b) {
        if ($a['severity'] === 'high' && $b['severity'] !== 'high') return -1;
        if ($b['severity'] === 'high' && $a['severity'] !== 'high') return 1;
        return $b['count'] - $a['count'];
    });
    
    return array_slice($cannibalization, 0, 20); // Top 20
}

// v3.2: Quick Wins - Recommendations for improvement
function wseo_get_quick_wins() {
    global $wpdb;
    
    $quick_wins = array(
        'missing_meta' => array(),
        'opportunities' => array(),
        'technical' => array(),
        'top_to_optimize' => array()
    );
    
    // v3.2: Excluded system pages - these don't need SEO optimization
    // Slugs to exclude
    $excluded_slugs = array(
        '404', 'error-404', 'page-404',
        'login', 'log-in', 'signin', 'sign-in', 'prihlasenie', 'prihlasit',
        'logout', 'log-out', 'signout', 'sign-out', 'odhlasenie', 'odhlasit',
        'register', 'registration', 'registracia', 'signup', 'sign-up',
        'lost-password', 'lostpassword', 'password-recovery', 'reset-password', 'zabudli-ste-heslo', 'zabudnute-heslo',
        'my-account', 'myaccount', 'account', 'moj-ucet', 'ucet',
        'cart', 'kosik', 'shopping-cart', 'nakupny-kosik',
        'checkout', 'pokladna', 'objednavka',
        'thank-you', 'thankyou', 'dakujeme', 'order-received',
        'submissions', 'form-submissions',
        'popup', 'popups', 'modal',
        'image-licenses', 'image-license', 'licencie-obrazkov',
        'privacy-policy', 'privacy', 'gdpr', 'ochrana-osobnych-udajov', 'sukromie',
        'terms', 'terms-and-conditions', 'terms-of-service', 'obchodne-podmienky', 'vop',
        'cookie-policy', 'cookies', 'cookie',
        'search', 'vyhladavanie',
        'wishlist', 'zoznam-zelani',
        // v3.2: Additional system pages
        'pridat-vyhodnu-ponuku', 'pridat-clanok', 'add-post', 'add-article', 'add-listing',
        'oblubene', 'favorites', 'favourites',
        'ptb-authors', 'authors', 'autori',
        'compare', 'porovnanie', 'porovnat',
        'sample-page', 'ukazka-stranky',
        'draft', 'koncept', 'test', 'testing',
        'maintenance', 'udrzba', 'coming-soon', 'pripravujeme',
        'sitemap', 'mapa-stranok',
        'feed', 'rss',
        'wp-admin', 'admin',
        'embed', 'oembed',
        'attachment', 'priloha'
    );
    $excluded_slugs_sql = "'" . implode("','", array_map('esc_sql', $excluded_slugs)) . "'";
    
    // Titles to exclude (partial match)
    $excluded_title_patterns = array(
        '%404%', '%login%', '%logout%', '%prihlás%', '%odhlás%',
        '%password%', '%heslo%', '%hesla%',
        '%registr%', '%account%', '%účet%', '%ucet%',
        '%cart%', '%košík%', '%kosik%',
        '%checkout%', '%pokladňa%', '%pokladna%',
        '%thank%you%', '%ďakujeme%', '%dakujeme%',
        '%submission%', '%popup%', '%modal%',
        '%license%', '%licenc%',
        '%privacy%', '%gdpr%', '%súkrom%', '%sukrom%',
        '%terms%', '%podmienk%', '%cookie%',
        '%wishlist%', '%želan%', '%zelan%',
        // v3.2: Additional title patterns
        '%pridať%článok%', '%pridat%clanok%', '%add%post%', '%add%article%',
        '%pridať%ponuk%', '%pridat%ponuk%', '%add%listing%', '%add%offer%',
        '%obľúben%', '%oblubene%', '%favorite%', '%favourite%',
        '%PTB %', '%author%', '%autori%',
        '%porovn%', '%compare%',
        '%sample%', '%ukážk%', '%ukazk%', '%test%',
        '%maintenance%', '%údržb%', '%udrzb%', '%coming soon%', '%pripravujeme%',
        '%sitemap%', '%mapa%stránok%',
        '%draft%', '%koncept%'
    );
    $title_exclude_conditions = array();
    foreach ($excluded_title_patterns as $pattern) {
        $title_exclude_conditions[] = "p.post_title NOT LIKE '" . esc_sql($pattern) . "'";
    }
    $title_exclude_sql = implode(' AND ', $title_exclude_conditions);
    
    // v3.2: System shortcodes to exclude - pages containing these are system pages
    $system_shortcodes = array(
        // WooCommerce
        '[woocommerce_cart]', '[woocommerce_checkout]', '[woocommerce_my_account]',
        '[woocommerce_order_tracking]', '[woocommerce_order_tracking]',
        '[shop_messages]', '[order_history]',
        '[recent_products]', '[featured_products]', '[products]',
        // User management
        '[login_form]', '[register_form]', '[password_form]', '[lost_password]',
        '[user_registration]', '[user_login]', '[user_profile]',
        '[custom_login_form]', '[custom_register]',
        // Membership/Subscriptions
        '[membership_login]', '[membership_register]', '[member_content]',
        '[subscription_details]', '[subscription_renewal]',
        // BuddyPress/BuddyBoss
        '[buddypress]', '[bp_register]', '[bp_login]',
        // Contact/Forms
        '[contact_form]', '[contact-form]', '[wpforms]', '[ninja_form]',
        '[formidable]', '[caldera_form]', '[gravityform]',
        // Listings/Directories
        '[submit_listing]', '[listing_dashboard]', '[listing_submission]',
        '[add_listing]', '[edit_listing]', '[user_dashboard]',
        '[vendor_dashboard]', '[seller_dashboard]',
        // Favorites/Wishlist
        '[favorites]', '[user_favorites]', '[wishlist]', '[yith_wcwl_wishlist]',
        // Search
        '[searchandfilter]', '[search_form]', '[ajax_search]',
        // Comparisons
        '[compare]', '[comparison_table]', '[yith_compare]',
        // Popups/Modals
        '[popup]', '[modal]', '[lightbox]',
        // System
        '[sitemap]', '[html_sitemap]', '[wp_sitemap]',
        '[authors]', '[author_box]',
        '[ptb_author]', '[ptb_taxonomies]',
        // Themes - Themify
        '[themify_author_box]', '[themify_builder]',
        // Themes - Flatsome  
        '[ux_account]', '[ux_wishlist]', '[ux_compare]',
        // Downloads
        '[download_history]', '[purchase_history]', '[edd_]',
        // Bookings
        '[booking_form]', '[booking_calendar]', '[appointment_form]'
    );
    
    // Build content exclusion conditions
    $content_exclude_conditions = array();
    foreach ($system_shortcodes as $shortcode) {
        $content_exclude_conditions[] = "p.post_content NOT LIKE '%" . esc_sql($shortcode) . "%'";
    }
    $content_exclude_sql = implode(' AND ', $content_exclude_conditions);
    
    // 1. MISSING META DATA
    // Correct meta keys: _wseo_meta_title, _wseo_meta_description, _wseo_meta_keywords
    
    // Posts without meta description (excluding system pages)
    $no_meta_desc = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wseo_meta_description'
        WHERE p.post_status = 'publish'
        AND p.post_type IN ('post', 'page', 'product')
        AND (pm.meta_value IS NULL OR pm.meta_value = '')
        AND p.post_name NOT IN ({$excluded_slugs_sql})
        AND {$title_exclude_sql}
        AND {$content_exclude_sql}
        ORDER BY p.post_date DESC
        LIMIT 50
    ");
    if (!empty($no_meta_desc)) {
        $quick_wins['missing_meta'][] = array(
            'type' => 'no_meta_desc',
            'count' => count($no_meta_desc),
            'label' => 'článkov/stránok bez meta popisu',
            'icon' => '📝',
            'severity' => 'warning',
            'items' => array_slice($no_meta_desc, 0, 10)
        );
    }
    
    // Posts without focus keyword (excluding system pages)
    $no_focus_kw = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wseo_meta_keywords'
        WHERE p.post_status = 'publish'
        AND p.post_type IN ('post', 'page')
        AND (pm.meta_value IS NULL OR pm.meta_value = '')
        AND p.post_name NOT IN ({$excluded_slugs_sql})
        AND {$title_exclude_sql}
        AND {$content_exclude_sql}
        ORDER BY p.post_date DESC
        LIMIT 50
    ");
    if (!empty($no_focus_kw)) {
        $quick_wins['missing_meta'][] = array(
            'type' => 'no_focus_kw',
            'count' => count($no_focus_kw),
            'label' => 'článkov bez focus keyword',
            'icon' => '🎯',
            'severity' => 'warning',
            'items' => array_slice($no_focus_kw, 0, 10)
        );
    }
    
    // 2. OPPORTUNITIES FOR IMPROVEMENT
    
    // Short content (less than 300 words) - excluding system pages
    // v3.4: Estimate word count in SQL instead of loading 200 full post_content blobs into PHP
    // MySQL approximation: word_count ≈ 1 + (spaces in stripped content)
    // Not perfect but fast and good enough for dashboard overview
    $short_content_raw = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type,
               (LENGTH(TRIM(p.post_content)) - LENGTH(REPLACE(TRIM(p.post_content), ' ', '')) + 1) as approx_words
        FROM {$wpdb->posts} p
        WHERE p.post_status = 'publish'
        AND p.post_type IN ('post', 'page')
        AND p.post_content != ''
        AND p.post_name NOT IN ({$excluded_slugs_sql})
        AND {$title_exclude_sql}
        AND {$content_exclude_sql}
        AND (LENGTH(TRIM(p.post_content)) - LENGTH(REPLACE(TRIM(p.post_content), ' ', '')) + 1) < 300
        ORDER BY (LENGTH(TRIM(p.post_content)) - LENGTH(REPLACE(TRIM(p.post_content), ' ', '')) + 1) ASC
        LIMIT 50
    ");
    
    $short_content = array();
    foreach ($short_content_raw as $post) {
        if (intval($post->approx_words) > 0) {
            $short_content[] = array(
                'ID' => $post->ID,
                'post_title' => $post->post_title,
                'post_type' => $post->post_type,
                'word_count' => intval($post->approx_words)
            );
        }
    }
    
    if (!empty($short_content)) {
        // v3.4: Already sorted by SQL ORDER BY approx_words ASC
        $quick_wins['opportunities'][] = array(
            'type' => 'short_content',
            'count' => count($short_content),
            'label' => 'článkov s krátkym obsahom (<300 slov)',
            'icon' => '📄',
            'severity' => 'info',
            'items' => array_slice($short_content, 0, 10)
        );
    }
    
    // Pages without internal links (orphans)
    $orphans = get_option('wseo_orphan_pages', array());
    if (!empty($orphans)) {
        $orphan_items = array();
        foreach (array_slice($orphans, 0, 10) as $orphan_id) {
            $orphan_post = get_post($orphan_id);
            if ($orphan_post) {
                $orphan_items[] = array(
                    'ID' => $orphan_id,
                    'post_title' => $orphan_post->post_title,
                    'post_type' => $orphan_post->post_type
                );
            }
        }
        if (!empty($orphan_items)) {
            $quick_wins['opportunities'][] = array(
                'type' => 'orphan_pages',
                'count' => count($orphans),
                'label' => 'stránok bez interných odkazov (siroty)',
                'icon' => '🔗',
                'severity' => 'warning',
                'items' => $orphan_items
            );
        }
    }
    
    // Images without alt text — v3.4: reuse cached count to avoid duplicate query
    // (same query already runs in wseo_calculate_health_score)
    static $cached_images_no_alt = null;
    if ($cached_images_no_alt === null) {
        $cached_images_no_alt = $wpdb->get_var("
            SELECT COUNT(*)
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_wp_attachment_image_alt'
            WHERE p.post_type = 'attachment'
            AND p.post_mime_type LIKE 'image/%'
            AND (pm.meta_value IS NULL OR pm.meta_value = '')
        ");
    }
    $images_no_alt = $cached_images_no_alt;
    if ($images_no_alt > 0) {
        $quick_wins['opportunities'][] = array(
            'type' => 'images_no_alt',
            'count' => intval($images_no_alt),
            'label' => 'obrázkov bez alt textu',
            'icon' => '🖼️',
            'severity' => 'warning',
            'can_autofix' => true,
            'items' => array()
        );
    }
    
    // 3. TECHNICAL ISSUES
    
    // Duplicate meta titles
    $duplicate_titles = $wpdb->get_results("
        SELECT pm.meta_value as title, COUNT(*) as cnt, GROUP_CONCAT(p.ID) as post_ids
        FROM {$wpdb->postmeta} pm
        JOIN {$wpdb->posts} p ON pm.post_id = p.ID
        WHERE pm.meta_key = '_wseo_meta_title'
        AND pm.meta_value != ''
        AND p.post_status = 'publish'
        GROUP BY pm.meta_value
        HAVING cnt > 1
        ORDER BY cnt DESC
        LIMIT 10
    ");
    if (!empty($duplicate_titles)) {
        $dup_items = array();
        foreach ($duplicate_titles as $dup) {
            $dup_items[] = array(
                'title' => $dup->title,
                'count' => $dup->cnt,
                'post_ids' => explode(',', $dup->post_ids)
            );
        }
        $quick_wins['technical'][] = array(
            'type' => 'duplicate_titles',
            'count' => count($duplicate_titles),
            'label' => 'duplicitných meta titulkov',
            'icon' => '⚠️',
            'severity' => 'error',
            'items' => $dup_items
        );
    }
    
    // Pages set to noindex but have content
    $noindex_with_content = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type, LENGTH(p.post_content) as content_length
        FROM {$wpdb->posts} p
        JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE pm.meta_key = '_wseo_noindex'
        AND pm.meta_value = '1'
        AND p.post_status = 'publish'
        AND LENGTH(p.post_content) > 500
        ORDER BY content_length DESC
        LIMIT 10
    ");
    if (!empty($noindex_with_content)) {
        $quick_wins['technical'][] = array(
            'type' => 'noindex_content',
            'count' => count($noindex_with_content),
            'label' => 'stránok s noindex ale kvalitným obsahom',
            'icon' => '🚫',
            'severity' => 'warning',
            'items' => $noindex_with_content
        );
    }
    
    // v3.3: Check Schema.org settings - areaServed for LocalBusiness
    $schema_type = get_option('wseo_schema_type', 'Organization');
    $area_served = get_option('wseo_schema_area_served', '');
    $is_local_business = !in_array($schema_type, array('Organization', 'Corporation', 'NewsMediaOrganization'));
    
    if ($is_local_business && empty($area_served)) {
        $quick_wins['technical'][] = array(
            'type' => 'missing_area_served',
            'count' => 1,
            'label' => 'chýba areaServed pre LocalBusiness',
            'icon' => '🗺️',
            'severity' => 'warning',
            'description' => 'Pre typ "' . esc_html($schema_type) . '" Google odporúča vyplniť oblasť pôsobenia (areaServed). Nastavte v Globálne nastavenia → Schema.org.',
            'link' => admin_url('admin.php?page=webstudio-seo-global'),
            'link_text' => 'Nastaviť areaServed'
        );
    }
    
    // v3.3: Check if LocalBusiness has address
    $street = get_option('wseo_schema_address_street', '');
    $city = get_option('wseo_schema_address_city', '');
    
    if ($is_local_business && empty($street) && empty($city)) {
        $quick_wins['technical'][] = array(
            'type' => 'missing_address',
            'count' => 1,
            'label' => 'chýba adresa pre LocalBusiness Schema',
            'icon' => '📍',
            'severity' => 'error',
            'description' => 'Pre typ "' . esc_html($schema_type) . '" je adresa povinná! Google penalizuje LocalBusiness bez adresy.',
            'link' => admin_url('admin.php?page=webstudio-seo-global'),
            'link_text' => 'Vyplniť adresu'
        );
    }
    
    // v3.3: Check if Organization has logo
    $logo = get_option('wseo_schema_logo', '');
    if (empty($logo)) {
        $quick_wins['technical'][] = array(
            'type' => 'missing_logo',
            'count' => 1,
            'label' => 'chýba logo v Schema.org',
            'icon' => '🖼️',
            'severity' => 'warning',
            'description' => 'Logo je odporúčané pre Google Knowledge Panel. Nastavte v Globálne nastavenia → Schema.org.',
            'link' => admin_url('admin.php?page=webstudio-seo-global'),
            'link_text' => 'Nastaviť logo'
        );
    }
    
    // v3.3: Check if has opening hours for LocalBusiness
    if ($is_local_business) {
        $has_hours = false;
        $days = array('mon', 'tue', 'wed', 'thu', 'fri', 'sat', 'sun');
        foreach ($days as $day) {
            if (get_option("wseo_schema_hours_{$day}_open", '') !== '') {
                $has_hours = true;
                break;
            }
        }
        
        if (!$has_hours) {
            $quick_wins['technical'][] = array(
                'type' => 'missing_hours',
                'count' => 1,
                'label' => 'chýbajú otváracie hodiny',
                'icon' => '🕐',
                'severity' => 'info',
                'description' => 'Otváracie hodiny pomáhajú Google zobraziť "Otvorené teraz" vo výsledkoch vyhľadávania.',
                'link' => admin_url('admin.php?page=webstudio-seo-global'),
                'link_text' => 'Nastaviť hodiny'
            );
        }
    }
    
    // v3.3: NAP Audit (Name, Address, Phone) - critical for local SEO
    $org_name = get_option('wseo_schema_name', '');
    $phone = get_option('wseo_schema_phone', '');
    $nap_issues = array();
    
    if (empty($org_name)) {
        $nap_issues[] = 'Názov firmy';
    }
    if ($is_local_business && empty($city)) {
        $nap_issues[] = 'Adresa (mesto)';
    }
    if ($is_local_business && empty($phone)) {
        $nap_issues[] = 'Telefón';
    }
    
    if (!empty($nap_issues)) {
        $quick_wins['technical'][] = array(
            'type' => 'nap_incomplete',
            'count' => count($nap_issues),
            'label' => 'neúplné NAP údaje (Name, Address, Phone)',
            'icon' => '📇',
            'severity' => 'error',
            'description' => 'Chýba: ' . implode(', ', $nap_issues) . '. NAP konzistencia je kritická pre Local SEO a Google Business Profile.',
            'link' => admin_url('admin.php?page=webstudio-seo-global'),
            'link_text' => 'Doplniť NAP'
        );
    }
    
    // v3.3: Check GPS coordinates for LocalBusiness
    $lat = get_option('wseo_schema_geo_lat', '');
    $lng = get_option('wseo_schema_geo_lng', '');
    
    if ($is_local_business && (empty($lat) || empty($lng))) {
        $quick_wins['technical'][] = array(
            'type' => 'missing_gps',
            'count' => 1,
            'label' => 'chýbajú GPS súradnice',
            'icon' => '🛰️',
            'severity' => 'warning',
            'description' => 'GPS súradnice zlepšujú presnosť v Google Maps a lokálnom vyhľadávaní. Tiež umožňujú GeoSitemap.',
            'link' => admin_url('admin.php?page=webstudio-seo-global'),
            'link_text' => 'Nastaviť GPS'
        );
    }
    
    // 4. TOP POSTS TO OPTIMIZE (biggest potential) - excluding system pages
    $posts_with_scores = $wpdb->get_results("
        SELECT p.ID, p.post_title, p.post_type, p.post_name,
               pm_desc.meta_value as has_desc,
               pm_kw.meta_value as has_kw,
               pm_title.meta_value as has_title
        FROM {$wpdb->posts} p
        LEFT JOIN {$wpdb->postmeta} pm_desc ON p.ID = pm_desc.post_id AND pm_desc.meta_key = '_wseo_meta_description'
        LEFT JOIN {$wpdb->postmeta} pm_kw ON p.ID = pm_kw.post_id AND pm_kw.meta_key = '_wseo_meta_keywords'
        LEFT JOIN {$wpdb->postmeta} pm_title ON p.ID = pm_title.post_id AND pm_title.meta_key = '_wseo_meta_title'
        WHERE p.post_status = 'publish'
        AND p.post_type IN ('post', 'page')
        AND p.post_name NOT IN ({$excluded_slugs_sql})
        AND {$title_exclude_sql}
        AND {$content_exclude_sql}
        ORDER BY p.post_date DESC
        LIMIT 100
    ");
    
    $optimization_candidates = array();
    foreach ($posts_with_scores as $post) {
        $potential_gain = 0;
        $missing = array();
        
        if (empty($post->has_title)) {
            $potential_gain += 10;
            $missing[] = 'meta title';
        }
        if (empty($post->has_desc)) {
            $potential_gain += 15;
            $missing[] = 'meta popis';
        }
        if (empty($post->has_kw)) {
            $potential_gain += 10;
            $missing[] = 'focus keyword';
        }
        
        if ($potential_gain >= 10) {
            $optimization_candidates[] = array(
                'ID' => $post->ID,
                'post_title' => $post->post_title,
                'post_type' => $post->post_type,
                'potential_gain' => $potential_gain,
                'missing' => $missing
            );
        }
    }
    
    // Sort by potential gain
    usort($optimization_candidates, function($a, $b) {
        return $b['potential_gain'] - $a['potential_gain'];
    });
    
    $quick_wins['top_to_optimize'] = array_slice($optimization_candidates, 0, 5);
    
    return $quick_wins;
}

// Get quick stats
function wseo_get_quick_stats() {
    global $wpdb;
    
    $stats = array();
    
    // Total posts
    $stats['posts'] = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type = 'post'");
    
    // Total pages
    $stats['pages'] = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type = 'page'");
    
    // Products (if WooCommerce)
    if (post_type_exists('product')) {
        $stats['products'] = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status = 'publish' AND post_type = 'product'");
    }
    
    // Posts with focus keyword - v3.2.0: Include all post types
    // Meta key is '_wseo_meta_keywords' (saved from Focus Keyword field in meta box)
    $focus_query = "
        SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE pm.meta_key = '_wseo_meta_keywords' 
        AND pm.meta_value IS NOT NULL 
        AND pm.meta_value != ''
        AND TRIM(pm.meta_value) != ''
        AND p.post_status = 'publish'
        AND p.post_type IN ('post', 'page', 'product')
    ";
    $focus_count = $wpdb->get_var($focus_query);
    $stats['with_focus'] = intval($focus_count);
    
    // Debug: Store query for troubleshooting (visible in HTML comment)
    $stats['_debug_focus_query'] = $focus_query;
    
    // Posts with meta description - v3.2: Fixed meta key
    $stats['with_desc'] = $wpdb->get_var("
        SELECT COUNT(DISTINCT p.ID) FROM {$wpdb->posts} p
        INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
        WHERE pm.meta_key = '_wseo_meta_description' AND pm.meta_value != ''
        AND p.post_status = 'publish'
    ") ?: 0;
    
    // Internal links count - v3.2: Count from link_map if exists, otherwise estimate from content
    $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$wpdb->prefix}wseo_link_map'");
    if ($table_exists) {
        $link_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}wseo_link_map WHERE status = 'ok'");
        if ($link_count > 0) {
            $stats['internal_links'] = intval($link_count);
        } else {
            // Table exists but empty - estimate from content
            $site_url = home_url();
            $sample_links = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) FROM {$wpdb->posts} 
                WHERE post_status = 'publish' 
                AND post_type IN ('post', 'page', 'product')
                AND (post_content LIKE %s OR post_content LIKE %s)
            ", '%href="' . $site_url . '%', '%href="/%'));
            $stats['internal_links'] = intval($sample_links);
        }
    } else {
        // No link_map table - estimate: count posts with internal links
        $site_url = home_url();
        $sample_links = $wpdb->get_var($wpdb->prepare("
            SELECT COUNT(*) FROM {$wpdb->posts} 
            WHERE post_status = 'publish' 
            AND post_type IN ('post', 'page', 'product')
            AND (post_content LIKE %s OR post_content LIKE %s)
        ", '%href="' . $site_url . '%', '%href="/%'));
        $stats['internal_links'] = intval($sample_links);
    }
    
    // Redirects count
    $redirects = get_option('wseo_redirects', array());
    $stats['redirects'] = is_array($redirects) ? count($redirects) : 0;
    
    // v3.2: Average readability score calculation
    // v3.4: Reduced sample from 50 to 20 — cached in transient, doesn't need large sample
    $sample_posts = $wpdb->get_results("
        SELECT ID, post_content, post_title
        FROM {$wpdb->posts}
        WHERE post_status = 'publish'
        AND post_type IN ('post', 'page')
        AND post_content != ''
        AND LENGTH(post_content) > 500
        ORDER BY post_date DESC
        LIMIT 20
    ");
    
    $readability_scores = array();
    $locale = get_locale();
    $lang_code = substr($locale, 0, 2);
    $long_threshold = in_array($lang_code, ['sk', 'cs', 'de', 'ru']) ? 25 : 20;
    
    foreach ($sample_posts as $sp) {
        $clean_text = strip_tags($sp->post_content);
        $clean_text = preg_replace('/\s+/', ' ', $clean_text);
        $clean_text = trim($clean_text);
        
        if (mb_strlen($clean_text) < 100) continue; // Skip very short content
        
        $words = preg_split('/\s+/', $clean_text, -1, PREG_SPLIT_NO_EMPTY);
        $word_count = count($words);
        if ($word_count < 50) continue; // Need enough words
        
        $sentences = preg_split('/[.!?]+/', $clean_text, -1, PREG_SPLIT_NO_EMPTY);
        $sentence_count = count(array_filter($sentences, function($s) { return mb_strlen(trim($s)) > 3; }));
        if ($sentence_count < 3) continue;
        
        // Simple Flesch calculation
        $syllable_count = 0;
        foreach ($words as $word) {
            $syllable_count += max(1, preg_match_all('/[aeiouyáéíóúýäöüàèìòùâêîôûаеёиоуыэюя]/iu', $word));
        }
        
        $flesch_coeffs = array(
            'sk' => array(206.835, 1.3, 60.1),
            'cs' => array(206.835, 1.3, 60.1),
            'en' => array(206.835, 1.015, 84.6),
            'de' => array(180.0, 1.0, 58.5)
        );
        $coeffs = isset($flesch_coeffs[$lang_code]) ? $flesch_coeffs[$lang_code] : $flesch_coeffs['en'];
        
        $flesch = $coeffs[0] - ($coeffs[1] * ($word_count / $sentence_count)) - ($coeffs[2] * ($syllable_count / $word_count));
        $flesch = max(0, min(100, round($flesch)));
        
        // Convert to simple readability score
        $avg_words_per_sentence = $word_count / $sentence_count;
        $score = 0;
        if ($flesch >= 60) $score += 50;
        elseif ($flesch >= 40) $score += 30;
        else $score += 10;
        
        if ($avg_words_per_sentence <= $long_threshold) $score += 50;
        elseif ($avg_words_per_sentence <= $long_threshold + 5) $score += 30;
        else $score += 10;
        
        $readability_scores[] = $score;
    }
    
    $stats['avg_readability'] = !empty($readability_scores) ? round(array_sum($readability_scores) / count($readability_scores)) : 0;
    $stats['readability_count'] = count($readability_scores);
    
    return $stats;
}

// v3.4: Transient cache for dashboard data — eliminates ~25 DB queries on each page load
// Cache key includes site URL hash to avoid conflicts in multisite
$cache_key = 'wseo_dashboard_data';
$cache_ttl = HOUR_IN_SECONDS; // 1 hour

// Force refresh if requested
if (isset($_GET['wseo_refresh_dashboard']) && wp_verify_nonce($_GET['_wpnonce'] ?? '', 'wseo_refresh_dashboard')) {
    delete_transient($cache_key);
}

$dashboard_data = get_transient($cache_key);

if ($dashboard_data === false) {
    $dashboard_start = microtime(true);
    
    $health = wseo_calculate_health_score();
    $cannibalization = wseo_detect_keyword_cannibalization();
    $quick_wins = wseo_get_quick_wins();
    $stats = wseo_get_quick_stats();
    
    $dashboard_data = array(
        'health' => $health,
        'cannibalization' => $cannibalization,
        'quick_wins' => $quick_wins,
        'stats' => $stats,
        'generated' => time(),
        'generation_time' => round((microtime(true) - $dashboard_start) * 1000) // ms
    );
    
    set_transient($cache_key, $dashboard_data, $cache_ttl);
    $from_cache = false;
} else {
    $from_cache = true;
}

$health = $dashboard_data['health'];
$cannibalization = $dashboard_data['cannibalization'];
$quick_wins = $dashboard_data['quick_wins'];
$stats = $dashboard_data['stats'];
$cache_age = time() - ($dashboard_data['generated'] ?? time());
$generation_time = $dashboard_data['generation_time'] ?? 0;

// Score color
$score_color = '#dc3545'; // red
if ($health['score'] >= 80) $score_color = '#28a745'; // green
elseif ($health['score'] >= 60) $score_color = '#ffc107'; // yellow
elseif ($health['score'] >= 40) $score_color = '#fd7e14'; // orange
?>

<div class="wrap wseo-admin">
    <h1 style="display: flex; align-items: center; gap: 12px;">
        📊 SEO Dashboard
        <?php
        $refresh_url = wp_nonce_url(
            add_query_arg('wseo_refresh_dashboard', '1', admin_url('admin.php?page=webstudio-seo-dashboard')),
            'wseo_refresh_dashboard'
        );
        ?>
        <a href="<?php echo esc_url($refresh_url); ?>" class="button button-small" title="Prepočítať SEO dáta" style="margin-left: auto;">
            🔄 Obnoviť dáta
        </a>
    </h1>
    <p class="wseo-subtitle" style="display: flex; align-items: center; gap: 10px;">
        Prehľad SEO zdravia vašej stránky
        <span style="font-size: 12px; color: #888; margin-left: auto;">
            <?php if ($from_cache): ?>
                📋 Z cache (<?php echo human_time_diff($dashboard_data['generated']); ?> staré)
            <?php else: ?>
                ⚡ Čerstvé dáta (<?php echo $generation_time; ?> ms)
            <?php endif; ?>
        </span>
    </p>
    
    <style>
        .wseo-dashboard-grid {
            display: grid;
            grid-template-columns: 300px 1fr;
            gap: 20px;
            margin-top: 20px;
        }
        @media (max-width: 1200px) {
            .wseo-dashboard-grid {
                grid-template-columns: 1fr;
            }
        }
        .wseo-score-card {
            background: #fff;
            border-radius: 12px;
            padding: 30px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
        }
        .wseo-score-circle {
            width: 180px;
            height: 180px;
            border-radius: 50%;
            margin: 0 auto 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 48px;
            font-weight: bold;
            color: #fff;
            position: relative;
        }
        .wseo-score-circle::after {
            content: '/100';
            font-size: 18px;
            position: absolute;
            bottom: 45px;
            opacity: 0.8;
        }
        .wseo-score-label {
            font-size: 18px;
            color: #666;
            margin-top: 10px;
        }
        .wseo-main-content {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }
        .wseo-dashboard-card {
            background: #fff;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 1px 4px rgba(0,0,0,0.08);
        }
        .wseo-dashboard-card h3 {
            margin: 0 0 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #f0f0f0;
        }
        .wseo-issue-list {
            list-style: none;
            padding: 0;
            margin: 0;
        }
        .wseo-issue-list li {
            padding: 10px 12px;
            margin-bottom: 8px;
            border-radius: 6px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        .wseo-issue-list li.issue {
            background: #fff5f5;
            border-left: 4px solid #dc3545;
        }
        .wseo-issue-list li.warning {
            background: #fffbeb;
            border-left: 4px solid #ffc107;
        }
        .wseo-issue-list li.passed {
            background: #f0fff4;
            border-left: 4px solid #28a745;
        }
        .wseo-issue-list li a {
            margin-left: auto;
            text-decoration: none;
            padding: 4px 12px;
            background: #f0f0f0;
            border-radius: 4px;
            font-size: 12px;
        }
        .wseo-issue-list li a:hover {
            background: #e0e0e0;
        }
        .wseo-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
            gap: 15px;
        }
        .wseo-stat-box {
            text-align: center;
            padding: 15px;
            background: #f8f9fa;
            border-radius: 8px;
        }
        .wseo-stat-box .number {
            font-size: 28px;
            font-weight: bold;
            color: #2271b1;
        }
        .wseo-stat-box .label {
            font-size: 12px;
            color: #666;
            margin-top: 5px;
        }
        .wseo-cannibalization-table {
            width: 100%;
            border-collapse: collapse;
        }
        .wseo-cannibalization-table th,
        .wseo-cannibalization-table td {
            padding: 10px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        .wseo-cannibalization-table th {
            background: #f8f9fa;
            font-weight: 600;
        }
        .wseo-cannibalization-table tr:hover {
            background: #f8f9fa;
        }
        .wseo-severity-high {
            color: #dc3545;
            font-weight: bold;
        }
        .wseo-severity-medium {
            color: #fd7e14;
        }
        .wseo-keyword-badge {
            display: inline-block;
            padding: 2px 8px;
            background: #e9ecef;
            border-radius: 4px;
            font-family: monospace;
            font-size: 13px;
        }
        .wseo-post-links {
            font-size: 12px;
            color: #666;
        }
        .wseo-post-links a {
            color: #2271b1;
            text-decoration: none;
        }
        .wseo-post-links a:hover {
            text-decoration: underline;
        }
        .wseo-focus-tag {
            display: inline-block;
            padding: 1px 6px;
            background: #dc3545;
            color: #fff;
            border-radius: 3px;
            font-size: 10px;
            margin-left: 5px;
        }
        .wseo-empty-state {
            text-align: center;
            padding: 40px;
            color: #666;
        }
        .wseo-empty-state .dashicons {
            font-size: 48px;
            width: 48px;
            height: 48px;
            color: #28a745;
        }
        details.wseo-passed-details {
            margin-top: 15px;
        }
        details.wseo-passed-details summary {
            cursor: pointer;
            color: #28a745;
            font-weight: 500;
        }
    </style>
    
    <div class="wseo-dashboard-grid">
        <!-- Left: Score Card -->
        <div class="wseo-score-card">
            <div class="wseo-score-circle" style="background: <?php echo $score_color; ?>;">
                <?php echo $health['score']; ?>
            </div>
            <div class="wseo-score-label">
                <?php
                if ($health['score'] >= 80) echo '🎉 Výborne!';
                elseif ($health['score'] >= 60) echo '👍 Dobre';
                elseif ($health['score'] >= 40) echo '⚠️ Potrebuje prácu';
                else echo '🚨 Kritické';
                ?>
            </div>
            
            <div class="wseo-stats-grid" style="margin-top: 25px;">
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['posts']); ?></div>
                    <div class="label">Článkov</div>
                </div>
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['pages']); ?></div>
                    <div class="label">Stránok</div>
                </div>
                <?php if (isset($stats['products'])): ?>
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['products']); ?></div>
                    <div class="label">Produktov</div>
                </div>
                <?php endif; ?>
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['with_focus']); ?></div>
                    <div class="label">S kľúč. slovom</div>
                </div>
                <!-- DEBUG WSEO v3.2: with_focus=<?php echo $stats['with_focus']; ?> -->
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['internal_links']); ?></div>
                    <div class="label">S int. odkazmi</div>
                </div>
                <div class="wseo-stat-box">
                    <div class="number"><?php echo number_format($stats['redirects']); ?></div>
                    <div class="label">Presmerovaní</div>
                </div>
                <div class="wseo-stat-box" style="background: <?php echo $stats['avg_readability'] >= 70 ? '#e8f5e9' : ($stats['avg_readability'] >= 50 ? '#fff3e0' : '#ffebee'); ?>;">
                    <div class="number"><?php echo $stats['avg_readability']; ?>%</div>
                    <div class="label">Čitateľnosť</div>
                </div>
            </div>
            
            <!-- v3.3: Local SEO Status -->
            <?php
            $schema_type = get_option('wseo_schema_type', 'Organization');
            $is_local = !in_array($schema_type, array('Organization', 'Corporation', 'NewsMediaOrganization'));
            $has_lat = get_option('wseo_schema_geo_lat', '');
            $has_lng = get_option('wseo_schema_geo_lng', '');
            $has_gps = !empty($has_lat) && !empty($has_lng);
            $locations_count = count(get_option('wseo_schema_locations', array()));
            
            if ($is_local):
            ?>
            <div style="margin-top: 20px; padding: 15px; background: linear-gradient(135deg, #e3f2fd 0%, #f3e5f5 100%); border-radius: 8px; border-left: 4px solid #1976d2;">
                <h4 style="margin: 0 0 10px; color: #1565c0;">📍 Local SEO Status</h4>
                <div style="display: flex; flex-wrap: wrap; gap: 15px; font-size: 13px;">
                    <span style="background: #fff; padding: 5px 10px; border-radius: 4px;">
                        <?php echo $has_gps ? '✅' : '❌'; ?> GPS súradnice
                    </span>
                    <span style="background: #fff; padding: 5px 10px; border-radius: 4px;">
                        <?php echo !empty(get_option('wseo_schema_phone', '')) ? '✅' : '❌'; ?> Telefón (NAP)
                    </span>
                    <span style="background: #fff; padding: 5px 10px; border-radius: 4px;">
                        <?php echo !empty(get_option('wseo_schema_address_city', '')) ? '✅' : '❌'; ?> Adresa (NAP)
                    </span>
                    <?php if ($locations_count > 0): ?>
                    <span style="background: #e8f5e9; padding: 5px 10px; border-radius: 4px;">
                        🏪 <?php echo $locations_count; ?> pobočiek
                    </span>
                    <?php endif; ?>
                    <?php if ($has_gps): ?>
                    <a href="<?php echo esc_url(home_url('/geo-sitemap.kml')); ?>" target="_blank" style="background: #1976d2; color: #fff; padding: 5px 10px; border-radius: 4px; text-decoration: none;">
                        🗺️ GeoSitemap (KML)
                    </a>
                    <?php endif; ?>
                </div>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Right: Main Content -->
        <div class="wseo-main-content">
            <!-- Issues & Warnings -->
            <div class="wseo-dashboard-card">
                <h3>🔍 SEO Kontrola</h3>
                
                <?php if (empty($health['issues']) && empty($health['warnings'])): ?>
                    <div class="wseo-empty-state">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <p><strong>Všetko v poriadku!</strong><br>Nenašli sa žiadne kritické problémy.</p>
                    </div>
                <?php else: ?>
                    <ul class="wseo-issue-list">
                        <?php foreach ($health['issues'] as $issue): ?>
                            <li class="issue">
                                <span>❌ <?php echo esc_html($issue['text']); ?></span>
                                <a href="<?php echo esc_url($issue['link']); ?>">Opraviť</a>
                            </li>
                        <?php endforeach; ?>
                        
                        <?php foreach ($health['warnings'] as $warning): ?>
                            <li class="warning">
                                <span>⚠️ <?php echo esc_html($warning['text']); ?></span>
                                <a href="<?php echo esc_url($warning['link']); ?>">Pozrieť</a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
                
                <?php if (!empty($health['passed'])): ?>
                    <details class="wseo-passed-details">
                        <summary>✅ <?php echo count($health['passed']); ?> položiek v poriadku</summary>
                        <ul class="wseo-issue-list" style="margin-top: 10px;">
                            <?php foreach ($health['passed'] as $passed): ?>
                                <li class="passed">
                                    <span>✅ <?php echo esc_html($passed); ?></span>
                                </li>
                            <?php endforeach; ?>
                        </ul>
                    </details>
                <?php endif; ?>
            </div>
            
            <!-- Quick Wins - Recommendations -->
            <div class="wseo-dashboard-card">
                <h3>🚀 Rýchle výhry - Odporúčania</h3>
                <p style="color: #666; margin-bottom: 15px;">
                    Najľahšie príležitosti na zlepšenie SEO vašej stránky.
                </p>
                
                <style>
                .wseo-quickwins-section {
                    margin-bottom: 20px;
                }
                .wseo-quickwins-section h4 {
                    margin: 0 0 10px;
                    padding: 8px 12px;
                    background: #f5f5f5;
                    border-radius: 6px;
                    font-size: 13px;
                    color: #1d2327;
                }
                .wseo-quickwins-list {
                    list-style: none;
                    margin: 0;
                    padding: 0;
                }
                .wseo-quickwins-item {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 10px 12px;
                    border: 1px solid #e2e4e7;
                    border-radius: 6px;
                    margin-bottom: 8px;
                    background: #fff;
                    transition: all 0.2s;
                }
                .wseo-quickwins-item:hover {
                    border-color: #2271b1;
                    box-shadow: 0 2px 4px rgba(0,0,0,0.05);
                }
                .wseo-quickwins-item.severity-error {
                    border-left: 3px solid #d63638;
                }
                .wseo-quickwins-item.severity-warning {
                    border-left: 3px solid #dba617;
                }
                .wseo-quickwins-item.severity-info {
                    border-left: 3px solid #2271b1;
                }
                .wseo-quickwins-info {
                    display: flex;
                    align-items: center;
                    gap: 10px;
                }
                .wseo-quickwins-icon {
                    font-size: 18px;
                }
                .wseo-quickwins-count {
                    display: inline-flex;
                    align-items: center;
                    justify-content: center;
                    min-width: 28px;
                    height: 28px;
                    padding: 0 8px;
                    background: #f0f0f1;
                    border-radius: 14px;
                    font-weight: 600;
                    font-size: 13px;
                }
                .wseo-quickwins-item.severity-error .wseo-quickwins-count {
                    background: #fcebec;
                    color: #d63638;
                }
                .wseo-quickwins-item.severity-warning .wseo-quickwins-count {
                    background: #fcf3e3;
                    color: #996800;
                }
                .wseo-quickwins-text {
                    flex: 1;
                    font-size: 13px;
                }
                .wseo-quickwins-action {
                    margin-left: 10px;
                }
                .wseo-quickwins-action .button {
                    padding: 0 12px;
                    height: 28px;
                    line-height: 26px;
                    font-size: 12px;
                }
                .wseo-quickwins-dropdown {
                    display: none;
                    padding: 10px;
                    background: #f9f9f9;
                    border-radius: 0 0 6px 6px;
                    margin-top: -9px;
                    margin-bottom: 8px;
                    border: 1px solid #e2e4e7;
                    border-top: none;
                }
                .wseo-quickwins-dropdown.open {
                    display: block;
                }
                .wseo-quickwins-dropdown-list {
                    list-style: none;
                    margin: 0;
                    padding: 0;
                    max-height: 200px;
                    overflow-y: auto;
                }
                .wseo-quickwins-dropdown-item {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 6px 8px;
                    font-size: 12px;
                    border-bottom: 1px solid #eee;
                }
                .wseo-quickwins-dropdown-item:last-child {
                    border-bottom: none;
                }
                .wseo-quickwins-dropdown-item a {
                    text-decoration: none;
                    color: #2271b1;
                }
                .wseo-quickwins-dropdown-item a:hover {
                    text-decoration: underline;
                }
                .wseo-top-optimize {
                    background: linear-gradient(135deg, #f0f6fc 0%, #e7f3ff 100%);
                    border-radius: 8px;
                    padding: 15px;
                }
                .wseo-top-optimize h4 {
                    margin: 0 0 12px;
                    color: #1d2327;
                }
                .wseo-top-optimize-item {
                    display: flex;
                    align-items: center;
                    justify-content: space-between;
                    padding: 8px 10px;
                    background: #fff;
                    border-radius: 6px;
                    margin-bottom: 6px;
                    font-size: 13px;
                }
                .wseo-top-optimize-item:last-child {
                    margin-bottom: 0;
                }
                .wseo-potential-badge {
                    background: #00a32a;
                    color: #fff;
                    padding: 2px 8px;
                    border-radius: 10px;
                    font-size: 11px;
                    font-weight: 600;
                }
                .wseo-empty-quickwins {
                    text-align: center;
                    padding: 30px;
                    color: #666;
                }
                .wseo-empty-quickwins .dashicons {
                    font-size: 48px;
                    width: 48px;
                    height: 48px;
                    color: #00a32a;
                    margin-bottom: 10px;
                }
                </style>
                
                <?php
                $has_any_issues = !empty($quick_wins['missing_meta']) || !empty($quick_wins['opportunities']) || !empty($quick_wins['technical']);
                
                if (!$has_any_issues && empty($quick_wins['top_to_optimize'])):
                ?>
                    <div class="wseo-empty-quickwins">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <p><strong>Výborne!</strong><br>Nenašli sme žiadne urgentné problémy na riešenie.</p>
                    </div>
                <?php else: ?>
                
                    <?php if (!empty($quick_wins['missing_meta'])): ?>
                    <div class="wseo-quickwins-section">
                        <h4>📝 Chýbajúce meta údaje</h4>
                        <ul class="wseo-quickwins-list">
                            <?php foreach ($quick_wins['missing_meta'] as $idx => $issue): ?>
                            <li>
                                <div class="wseo-quickwins-item severity-<?php echo esc_attr($issue['severity']); ?>" data-dropdown="meta-<?php echo $idx; ?>">
                                    <div class="wseo-quickwins-info">
                                        <span class="wseo-quickwins-icon"><?php echo $issue['icon']; ?></span>
                                        <span class="wseo-quickwins-count"><?php echo $issue['count']; ?></span>
                                        <span class="wseo-quickwins-text"><?php echo esc_html($issue['label']); ?></span>
                                    </div>
                                    <div class="wseo-quickwins-action">
                                        <button type="button" class="button wseo-toggle-dropdown" data-target="meta-<?php echo $idx; ?>">
                                            Zobraziť ▼
                                        </button>
                                    </div>
                                </div>
                                <?php if (!empty($issue['items'])): ?>
                                <div class="wseo-quickwins-dropdown" id="dropdown-meta-<?php echo $idx; ?>">
                                    <ul class="wseo-quickwins-dropdown-list">
                                        <?php foreach ($issue['items'] as $item): ?>
                                        <li class="wseo-quickwins-dropdown-item">
                                            <span><?php echo esc_html(mb_substr($item->post_title, 0, 50)); ?><?php echo mb_strlen($item->post_title) > 50 ? '...' : ''; ?></span>
                                            <a href="<?php echo get_edit_post_link($item->ID); ?>">Upraviť →</a>
                                        </li>
                                        <?php endforeach; ?>
                                        <?php if ($issue['count'] > 10): ?>
                                        <li class="wseo-quickwins-dropdown-item" style="color: #666; font-style: italic;">
                                            ... a <?php echo $issue['count'] - 10; ?> ďalších
                                        </li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                                <?php endif; ?>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($quick_wins['opportunities'])): ?>
                    <div class="wseo-quickwins-section">
                        <h4>📈 Príležitosti na zlepšenie</h4>
                        <ul class="wseo-quickwins-list">
                            <?php foreach ($quick_wins['opportunities'] as $idx => $issue): ?>
                            <li>
                                <div class="wseo-quickwins-item severity-<?php echo esc_attr($issue['severity']); ?>" data-dropdown="opp-<?php echo $idx; ?>">
                                    <div class="wseo-quickwins-info">
                                        <span class="wseo-quickwins-icon"><?php echo $issue['icon']; ?></span>
                                        <span class="wseo-quickwins-count"><?php echo $issue['count']; ?></span>
                                        <span class="wseo-quickwins-text"><?php echo esc_html($issue['label']); ?></span>
                                    </div>
                                    <div class="wseo-quickwins-action">
                                        <?php if (!empty($issue['can_autofix'])): ?>
                                        <a href="<?php echo admin_url('admin.php?page=webstudio-seo-images'); ?>" class="button button-primary" style="margin-right: 5px;">Auto-fix</a>
                                        <?php endif; ?>
                                        <?php if (!empty($issue['items'])): ?>
                                        <button type="button" class="button wseo-toggle-dropdown" data-target="opp-<?php echo $idx; ?>">
                                            Zobraziť ▼
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php if (!empty($issue['items'])): ?>
                                <div class="wseo-quickwins-dropdown" id="dropdown-opp-<?php echo $idx; ?>">
                                    <ul class="wseo-quickwins-dropdown-list">
                                        <?php foreach ($issue['items'] as $item): ?>
                                        <li class="wseo-quickwins-dropdown-item">
                                            <span>
                                                <?php echo esc_html(mb_substr($item['post_title'], 0, 40)); ?><?php echo mb_strlen($item['post_title']) > 40 ? '...' : ''; ?>
                                                <?php if (isset($item['word_count'])): ?>
                                                <small style="color: #999;">(<?php echo $item['word_count']; ?> slov)</small>
                                                <?php endif; ?>
                                            </span>
                                            <a href="<?php echo get_edit_post_link($item['ID']); ?>">Upraviť →</a>
                                        </li>
                                        <?php endforeach; ?>
                                        <?php if ($issue['count'] > 10): ?>
                                        <li class="wseo-quickwins-dropdown-item" style="color: #666; font-style: italic;">
                                            ... a <?php echo $issue['count'] - 10; ?> ďalších
                                        </li>
                                        <?php endif; ?>
                                    </ul>
                                </div>
                                <?php endif; ?>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($quick_wins['technical'])): ?>
                    <div class="wseo-quickwins-section">
                        <h4>⚡ Technické problémy</h4>
                        <ul class="wseo-quickwins-list">
                            <?php foreach ($quick_wins['technical'] as $idx => $issue): ?>
                            <li>
                                <div class="wseo-quickwins-item severity-<?php echo esc_attr($issue['severity']); ?>">
                                    <div class="wseo-quickwins-info">
                                        <span class="wseo-quickwins-icon"><?php echo $issue['icon']; ?></span>
                                        <span class="wseo-quickwins-count"><?php echo $issue['count']; ?></span>
                                        <span class="wseo-quickwins-text"><?php echo esc_html($issue['label']); ?></span>
                                    </div>
                                    <div class="wseo-quickwins-action">
                                        <?php if (!empty($issue['link'])): ?>
                                        <a href="<?php echo esc_url($issue['link']); ?>" class="button button-small">
                                            <?php echo esc_html($issue['link_text'] ?? 'Opraviť'); ?> →
                                        </a>
                                        <?php elseif (!empty($issue['items'])): ?>
                                        <button type="button" class="button wseo-toggle-dropdown" data-target="tech-<?php echo $idx; ?>">
                                            Zobraziť ▼
                                        </button>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <?php if (!empty($issue['description'])): ?>
                                <div style="margin: 8px 0 0 32px; color: #666; font-size: 12px; line-height: 1.4;">
                                    <?php echo esc_html($issue['description']); ?>
                                </div>
                                <?php endif; ?>
                                <?php if (!empty($issue['items'])): ?>
                                <div class="wseo-quickwins-dropdown" id="dropdown-tech-<?php echo $idx; ?>">
                                    <ul class="wseo-quickwins-dropdown-list">
                                        <?php foreach ($issue['items'] as $item): 
                                            if ($issue['type'] === 'duplicate_titles'): ?>
                                        <li class="wseo-quickwins-dropdown-item">
                                            <span>"<?php echo esc_html(mb_substr($item['title'], 0, 35)); ?>..." <small style="color: #d63638;">(<?php echo $item['count']; ?>× duplicitný)</small></span>
                                        </li>
                                            <?php else: ?>
                                        <li class="wseo-quickwins-dropdown-item">
                                            <span><?php echo esc_html(mb_substr($item->post_title, 0, 40)); ?><?php echo mb_strlen($item->post_title) > 40 ? '...' : ''; ?></span>
                                            <a href="<?php echo get_edit_post_link($item->ID); ?>">Upraviť →</a>
                                        </li>
                                            <?php endif; ?>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                                <?php endif; ?>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($quick_wins['top_to_optimize'])): ?>
                    <div class="wseo-top-optimize">
                        <h4>💡 Top 5 článkov na optimalizáciu</h4>
                        <?php foreach ($quick_wins['top_to_optimize'] as $idx => $item): ?>
                        <div class="wseo-top-optimize-item">
                            <span>
                                <strong><?php echo $idx + 1; ?>.</strong>
                                <?php echo esc_html(mb_substr($item['post_title'], 0, 35)); ?><?php echo mb_strlen($item['post_title']) > 35 ? '...' : ''; ?>
                                <small style="color: #666;">(chýba: <?php echo implode(', ', $item['missing']); ?>)</small>
                            </span>
                            <span>
                                <span class="wseo-potential-badge">+<?php echo $item['potential_gain']; ?> bodov</span>
                                <a href="<?php echo get_edit_post_link($item['ID']); ?>" class="button button-small" style="margin-left: 8px;">Optimalizovať</a>
                            </span>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    <?php endif; ?>
                    
                <?php endif; ?>
                
                <script>
                jQuery(function($) {
                    $('.wseo-toggle-dropdown').on('click', function() {
                        var target = $(this).data('target');
                        var $dropdown = $('#dropdown-' + target);
                        $dropdown.toggleClass('open');
                        $(this).text($dropdown.hasClass('open') ? 'Skryť ▲' : 'Zobraziť ▼');
                    });
                });
                </script>
            </div>
            
            <!-- Keyword Cannibalization -->
            <div class="wseo-dashboard-card">
                <h3>🔄 Keyword Cannibalization</h3>
                <p style="color: #666; margin-bottom: 15px;">
                    Stránky, ktoré súťažia o rovnaké kľúčové slová vo vyhľadávačoch.
                </p>
                
                <?php if (empty($cannibalization)): ?>
                    <div class="wseo-empty-state">
                        <span class="dashicons dashicons-yes-alt"></span>
                        <p><strong>Žiadna kanibalizácia!</strong><br>Vaše stránky nesúťažia o rovnaké kľúčové slová.</p>
                    </div>
                <?php else: ?>
                    <style>
                    .wseo-cannibal-accordion {
                        border: 1px solid #ddd;
                        border-radius: 8px;
                        overflow: hidden;
                    }
                    .wseo-cannibal-item {
                        border-bottom: 1px solid #eee;
                    }
                    .wseo-cannibal-item:last-child {
                        border-bottom: none;
                    }
                    .wseo-cannibal-header {
                        display: flex;
                        align-items: center;
                        justify-content: space-between;
                        padding: 12px 15px;
                        background: #f9f9f9;
                        cursor: pointer;
                        transition: background 0.2s;
                    }
                    .wseo-cannibal-header:hover {
                        background: #f0f0f0;
                    }
                    .wseo-cannibal-header.open {
                        background: #e8f4fc;
                        border-bottom: 1px solid #ddd;
                    }
                    .wseo-cannibal-keyword {
                        display: flex;
                        align-items: center;
                        gap: 10px;
                    }
                    .wseo-cannibal-keyword-text {
                        font-weight: 600;
                        font-size: 14px;
                    }
                    .wseo-cannibal-count {
                        background: #d63638;
                        color: #fff;
                        padding: 3px 10px;
                        border-radius: 12px;
                        font-size: 12px;
                        font-weight: 600;
                    }
                    .wseo-cannibal-count.medium {
                        background: #dba617;
                    }
                    .wseo-cannibal-arrow {
                        transition: transform 0.2s;
                        color: #666;
                    }
                    .wseo-cannibal-header.open .wseo-cannibal-arrow {
                        transform: rotate(180deg);
                    }
                    .wseo-cannibal-content {
                        display: none;
                        padding: 0;
                        background: #fff;
                    }
                    .wseo-cannibal-content.open {
                        display: block;
                    }
                    .wseo-cannibal-url {
                        display: flex;
                        align-items: center;
                        justify-content: space-between;
                        padding: 10px 15px;
                        border-bottom: 1px solid #f0f0f0;
                    }
                    .wseo-cannibal-url:last-child {
                        border-bottom: none;
                    }
                    .wseo-cannibal-url:hover {
                        background: #fafafa;
                    }
                    .wseo-cannibal-url-title {
                        display: flex;
                        align-items: center;
                        gap: 8px;
                        flex: 1;
                    }
                    .wseo-cannibal-url-title a {
                        color: #2271b1;
                        text-decoration: none;
                        font-size: 13px;
                    }
                    .wseo-cannibal-url-title a:hover {
                        text-decoration: underline;
                    }
                    .wseo-cannibal-focus {
                        background: #e8f5e9;
                        color: #2e7d32;
                        padding: 2px 8px;
                        border-radius: 10px;
                        font-size: 10px;
                        font-weight: 600;
                    }
                    .wseo-cannibal-type {
                        background: #f0f0f0;
                        padding: 2px 8px;
                        border-radius: 10px;
                        font-size: 11px;
                        color: #666;
                    }
                    .wseo-cannibal-edit {
                        font-size: 12px;
                        color: #2271b1;
                        text-decoration: none;
                    }
                    </style>
                    
                    <div class="wseo-cannibal-accordion">
                        <?php foreach ($cannibalization as $idx => $item): ?>
                        <div class="wseo-cannibal-item">
                            <div class="wseo-cannibal-header" onclick="this.classList.toggle('open'); this.nextElementSibling.classList.toggle('open');">
                                <div class="wseo-cannibal-keyword">
                                    <span class="wseo-cannibal-keyword-text"><?php echo esc_html($item['keyword']); ?></span>
                                    <span class="wseo-cannibal-count <?php echo $item['severity'] === 'high' ? '' : 'medium'; ?>">
                                        <?php echo $item['count']; ?> stránok
                                    </span>
                                </div>
                                <span class="wseo-cannibal-arrow dashicons dashicons-arrow-down-alt2"></span>
                            </div>
                            <div class="wseo-cannibal-content">
                                <?php foreach ($item['posts'] as $post): ?>
                                <div class="wseo-cannibal-url">
                                    <div class="wseo-cannibal-url-title">
                                        <a href="<?php echo get_edit_post_link($post['id']); ?>">
                                            <?php echo esc_html($post['title']); ?>
                                        </a>
                                        <?php if ($post['is_focus']): ?>
                                        <span class="wseo-cannibal-focus">FOCUS KW</span>
                                        <?php endif; ?>
                                        <span class="wseo-cannibal-type"><?php 
                                            $type = get_post_type($post['id']);
                                            echo $type === 'post' ? 'Článok' : ($type === 'page' ? 'Stránka' : ucfirst($type));
                                        ?></span>
                                    </div>
                                    <a href="<?php echo get_edit_post_link($post['id']); ?>" class="wseo-cannibal-edit">Upraviť →</a>
                                </div>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <p style="margin-top: 15px; font-size: 13px; color: #666;">
                        💡 <strong>Tip:</strong> Pre každé kľúčové slovo by mala byť jedna hlavná stránka. 
                        Ostatné stránky prepíšte na iné kľúčové slová alebo ich zlúčte.
                    </p>
                <?php endif; ?>
            </div>
            
            <!-- Quick Actions -->
            <div class="wseo-dashboard-card">
                <h3>⚡ Rýchle akcie</h3>
                <div style="display: flex; gap: 10px; flex-wrap: wrap;">
                    <a href="<?php echo admin_url('admin.php?page=webstudio-seo-global'); ?>" class="button">
                        ⚙️ Globálne nastavenia
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=webstudio-seo-images'); ?>" class="button">
                        🖼️ Optimalizovať obrázky
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=webstudio-seo-links'); ?>" class="button">
                        🔗 Interné odkazy
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=webstudio-seo-redirects'); ?>" class="button">
                        🔄 Presmerovania
                    </a>
                    <a href="<?php echo home_url('/sitemap.xml'); ?>" class="button" target="_blank">
                        🗺️ Zobraziť Sitemap
                    </a>
                    <a href="https://search.google.com/test/rich-results" class="button" target="_blank">
                        📊 Test Rich Results
                    </a>
                </div>
            </div>
        </div>
    </div>
    
    <p style="margin-top: 30px; color: #999; font-size: 12px;">
        WebStudio SEO Pro v<?php echo WSEO_VERSION; ?> | 
        Posledná kontrola: <?php echo date('d.m.Y H:i'); ?>
    </p>
</div>
