/**
 * Intelligent Page Prefetching v3.4
 * WebStudio SEO Pro
 * © 2026 Karol Kameniczki - Webstudio.ltd
 * 
 * v3.4 CHANGES:
 * - IntersectionObserver as PRIMARY strategy (viewport-based, zero TBT)
 * - Hover as SUPPLEMENTARY strategy (for links above fold already visible)
 * - Touch intent for mobile (prefetch on touchstart)
 * - requestIdleCallback wrapper (non-blocking initialization)
 * - Save-Data / slow connection respected
 * - Completely disabled for logged-in users
 * - Not loaded on WooCommerce cart/checkout/account/product (server-side)
 */
(function() {
    'use strict';
    
    // ========================================
    // SAFETY CHECKS - Exit immediately if:
    // 1. User is logged in (admin bar)
    // 2. Page builder is active
    // 3. Admin/preview mode
    // ========================================
    
    function shouldDisable() {
        var url = window.location.href;
        if (url.indexOf('/wp-admin') !== -1 ||
            url.indexOf('wp-login') !== -1 ||
            url.indexOf('elementor-preview') !== -1 ||
            url.indexOf('tb-preview') !== -1 ||
            url.indexOf('fl_builder') !== -1 ||
            url.indexOf('et_fb=1') !== -1 ||
            url.indexOf('bricks=') !== -1 ||
            url.indexOf('ct_builder') !== -1 ||
            url.indexOf('vc_editable') !== -1 ||
            url.indexOf('preview=true') !== -1 ||
            url.indexOf('preview_id=') !== -1) {
            return true;
        }
        
        if (document.getElementById('wpadminbar')) return true;
        
        var html = document.documentElement;
        if (html.classList.contains('wp-toolbar')) return true;
        
        var body = document.body;
        if (body) {
            if (body.classList.contains('logged-in') ||
                body.classList.contains('admin-bar') ||
                body.classList.contains('wp-admin') ||
                body.classList.contains('elementor-editor-active') ||
                body.classList.contains('themify_builder_active') ||
                body.classList.contains('tb-editor-active') ||
                body.classList.contains('fl-builder-edit') ||
                body.classList.contains('et-fb') ||
                body.classList.contains('bricks-is-frontend') ||
                body.classList.contains('oxygen-builder-body') ||
                body.classList.contains('vc_editor')) {
                return true;
            }
        }
        
        if (window.elementor || window.elementorFrontend ||
            window.ThemifyBuilder || window.themifyBuilder ||
            window.FLBuilder || window.ET_Builder ||
            window.bricksData || window.oxygen ||
            window.vc || (window.wp && window.wp.customize)) {
            return true;
        }
        
        return false;
    }
    
    // ========================================
    // MAIN LOGIC - Only runs for visitors
    // ========================================
    
    var config = window.wseoConfig || {
        prefetchEnabled: true,
        prefetchStrategy: 'viewport',
        prefetchHoverDelay: 250,
        prefetchExclude: [],
        prefetchMaxConcurrent: 1,
        prefetchMaxPages: 5
    };
    
    var prefetchedUrls = {};
    var currentPrefetches = 0;
    var maxPrefetches = config.prefetchMaxConcurrent || 1;
    var maxPages = config.prefetchMaxPages || 5;
    var prefetchCount = 0;
    
    /**
     * Check if URL is valid for prefetch
     */
    function isValidUrl(url) {
        if (!url) return false;
        
        try {
            var urlObj = new URL(url, window.location.origin);
            
            if (urlObj.origin !== window.location.origin) return false;
            if (urlObj.pathname === window.location.pathname) return false;
            if (urlObj.pathname.indexOf('/wp-admin') !== -1) return false;
            if (urlObj.pathname.indexOf('/wp-login') !== -1) return false;
            if (prefetchedUrls[urlObj.pathname]) return false;
            
            for (var i = 0; i < config.prefetchExclude.length; i++) {
                try {
                    if (new RegExp(config.prefetchExclude[i]).test(url)) return false;
                } catch (e) {}
            }
            
            return true;
        } catch (e) {
            return false;
        }
    }
    
    /**
     * Check if connection supports prefetch
     * Respects Save-Data header and slow connections
     */
    function canPrefetch() {
        if ('connection' in navigator) {
            var conn = navigator.connection || navigator.mozConnection || navigator.webkitConnection;
            if (conn) {
                if (conn.saveData) return false;
                var type = conn.effectiveType;
                if (type === 'slow-2g' || type === '2g') return false;
            }
        }
        return true;
    }
    
    /**
     * Prefetch a URL using native <link rel="prefetch">
     * Low priority by design - browser handles scheduling
     */
    function prefetchUrl(url) {
        if (!isValidUrl(url)) return;
        if (currentPrefetches >= maxPrefetches) return;
        if (prefetchCount >= maxPages) return;
        if (!canPrefetch()) return;
        
        try {
            var urlObj = new URL(url, window.location.origin);
            prefetchedUrls[urlObj.pathname] = true;
        } catch (e) {
            prefetchedUrls[url] = true;
        }
        
        currentPrefetches++;
        prefetchCount++;
        
        var link = document.createElement('link');
        link.rel = 'prefetch';
        link.href = url;
        link.as = 'document';
        
        link.onload = function() { currentPrefetches--; };
        link.onerror = function() {
            currentPrefetches--;
            try {
                var u = new URL(url, window.location.origin);
                delete prefetchedUrls[u.pathname];
            } catch (e) {
                delete prefetchedUrls[url];
            }
            prefetchCount--;
        };
        
        document.head.appendChild(link);
    }
    
    /**
     * v3.4: PRIMARY strategy - IntersectionObserver
     * Prefetches links as they scroll into viewport
     * Zero TBT impact - browser-native, passive observation
     */
    function initViewportPrefetch() {
        if (!('IntersectionObserver' in window)) {
            // Fallback for old browsers: hover only
            initHoverPrefetch();
            return;
        }
        
        var observer = new IntersectionObserver(function(entries) {
            for (var i = 0; i < entries.length; i++) {
                if (entries[i].isIntersecting && prefetchCount < maxPages) {
                    prefetchUrl(entries[i].target.href);
                    observer.unobserve(entries[i].target);
                }
            }
        }, {
            // 200px ahead = browser starts fetching before user reaches the link
            rootMargin: '200px 0px',
            threshold: 0
        });
        
        // Scoped to content areas to avoid nav/footer overhead
        var contentSelectors = [
            'main a[href]',
            'article a[href]',
            '.entry-content a[href]',
            '.post-content a[href]',
            '.page-content a[href]',
            '#content a[href]',
            '.site-content a[href]',
            '.elementor-widget-container a[href]'
        ];
        
        var observed = {};
        var origin = window.location.origin;
        
        // Try content-scoped selectors first
        var links = document.querySelectorAll(contentSelectors.join(', '));
        
        // Fallback: if no content links found, observe all internal links
        if (links.length === 0) {
            links = document.querySelectorAll('a[href^="/"], a[href^="' + origin + '"]');
        }
        
        // Limit observed links to 3x max pages (avoid tracking hundreds of links)
        var limit = maxPages * 3;
        for (var i = 0; i < links.length && Object.keys(observed).length < limit; i++) {
            var href = links[i].href;
            if (isValidUrl(href) && !observed[href]) {
                observed[href] = true;
                observer.observe(links[i]);
            }
        }
    }
    
    /**
     * v3.4: SUPPLEMENTARY strategy - Hover intent
     * Catches links user is about to click (above fold, already visible)
     * Uses event delegation for minimal memory overhead
     */
    function initHoverPrefetch() {
        var hoverTimer;
        var delay = config.prefetchHoverDelay || 250;
        var currentLink = null;
        
        document.addEventListener('mouseover', function(e) {
            var link = e.target.closest ? e.target.closest('a[href]') : null;
            if (!link || link === currentLink) return;
            
            currentLink = link;
            clearTimeout(hoverTimer);
            hoverTimer = setTimeout(function() {
                prefetchUrl(link.href);
            }, delay);
        }, { passive: true });
        
        document.addEventListener('mouseout', function(e) {
            var link = e.target.closest ? e.target.closest('a[href]') : null;
            if (link === currentLink) {
                clearTimeout(hoverTimer);
                currentLink = null;
            }
        }, { passive: true });
    }
    
    /**
     * v3.4: Touch intent for mobile
     * ~300ms between touchstart and navigation = time to start prefetch
     */
    function initTouchPrefetch() {
        document.addEventListener('touchstart', function(e) {
            var link = e.target.closest ? e.target.closest('a[href]') : null;
            if (link) {
                prefetchUrl(link.href);
            }
        }, { passive: true });
    }
    
    /**
     * Initialize prefetching
     */
    function init() {
        if (shouldDisable()) return;
        if (!config.prefetchEnabled) return;
        if (!canPrefetch()) return;
        
        var strategy = config.prefetchStrategy || 'viewport';
        
        if (strategy === 'hover') {
            // Legacy mode: hover only (user explicitly chose this in settings)
            initHoverPrefetch();
        } else {
            // v3.4 default: viewport (IO) + hover + touch
            initViewportPrefetch();
            initHoverPrefetch();
            
            if ('ontouchstart' in window) {
                initTouchPrefetch();
            }
        }
    }
    
    // ========================================
    // START - Non-blocking initialization
    // v3.4: requestIdleCallback = zero TBT impact
    // ========================================
    
    function scheduleInit() {
        if ('requestIdleCallback' in window) {
            requestIdleCallback(init, { timeout: 2000 });
        } else {
            setTimeout(init, 100);
        }
    }
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', scheduleInit);
    } else {
        scheduleInit();
    }
    
})();
