<?php
/**
 * Split Key License Management v3.0
 * 
 * Bezpečnostný systém s rozdelením kľúča:
 * - Server Half: Overovaný cez API
 * - Domain Half: Generovaný na serveri (SALT nikdy nie je tu!)
 * - Bez oboch polovíc = nefunguje
 * 
 * @package WebStudio_SEO_Pro
 * @version 3.0.0
 */

if (!defined('ABSPATH')) exit;

class WSEO_License {
    
    private $api_url = 'https://webstudio.ltd/wp-json/wseo-license/v1/validate';
    private $ping_url = 'https://webstudio.ltd/wp-json/wseo-license/v1/ping';
    private $trial_days = 14;
    
    // Verzia licenčného systému
    private $license_system_version = '3.0';
    
    public function __construct() {
        $this->track_first_activation();
        
        add_action('admin_init', [$this, 'check_license']);
        add_action('admin_init', [$this, 'handle_license_actions']);
        add_action('admin_notices', [$this, 'license_notices']);
        add_action('admin_init', [$this, 'maybe_send_telemetry']);
    }
    
    /**
     * Send telemetry ping (once per day)
     */
    public function maybe_send_telemetry() {
        $last_ping = get_option('wseo_last_telemetry_ping', 0);
        
        if (time() - $last_ping < DAY_IN_SECONDS) {
            return;
        }
        
        // v3.3.1: Determine license status for telemetry
        $license_key = get_option('wseo_license_key', '');
        $cached_valid = get_transient('wseo_license_valid');
        
        if (!empty($license_key) && $cached_valid === 'yes') {
            $license_status = 'licensed';
        } elseif ($this->is_trial_active()) {
            $license_status = 'trial';
        } else {
            $license_status = 'demo';
        }
        
        wp_remote_post($this->ping_url, [
            'timeout' => 5,
            'blocking' => false,
            'body' => [
                'domain' => $this->get_clean_domain(),
                'site_name' => get_bloginfo('name'),
                'plugin_version' => WSEO_VERSION,
                'wp_version' => get_bloginfo('version'),
                'php_version' => PHP_VERSION,
                'locale' => get_locale(),
                'active_theme' => get_template(),
                'is_multisite' => is_multisite() ? 1 : 0,
                'license_system' => $this->license_system_version,
                'license_status' => $license_status // v3.3.1: Send actual license status
            ]
        ]);
        
        update_option('wseo_last_telemetry_ping', time());
    }
    
    /**
     * Get clean domain (without www.)
     */
    private function get_clean_domain() {
        $domain = parse_url(home_url(), PHP_URL_HOST);
        return strtolower(preg_replace('/^www\./', '', $domain));
    }
    
    /**
     * Handle license form actions
     */
    public function handle_license_actions() {
        // Manual license check
        if (isset($_POST['wseo_check_license']) && wp_verify_nonce($_POST['_wpnonce'], 'wseo_check_license')) {
            delete_transient('wseo_license_valid');
            $this->validate_remote();
            wp_redirect(admin_url('admin.php?page=webstudio-seo-license&checked=1'));
            exit;
        }
        
        // Activate license
        if (isset($_POST['wseo_activate']) && check_admin_referer('wseo_activate_license')) {
            $new_key = strtoupper(sanitize_text_field($_POST['wseo_license_key']));
            
            // Validuj formát (WSEO-XXXX-XXXX-XXXX-XXXX)
            if (!preg_match('/^WSEO-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}-[A-Z0-9]{4}$/', $new_key)) {
                add_settings_error('wseo_license', 'invalid_format', 
                    '❌ Neplatný formát kľúča. Očakávaný formát: WSEO-XXXX-XXXX-XXXX-XXXX', 'error');
                return;
            }
            
            update_option('wseo_license_key', $new_key);
            delete_transient('wseo_license_valid');
            
            // Okamžite over
            $valid = $this->validate_remote();
            
            if ($valid) {
                add_settings_error('wseo_license', 'activated', 
                    '✅ Licencia úspešne aktivovaná!', 'success');
            }
        }
        
        // Deactivate license
        if (isset($_POST['wseo_deactivate']) && check_admin_referer('wseo_deactivate_license')) {
            delete_option('wseo_license_key');
            delete_option('wseo_license_expires');
            delete_option('wseo_license_domain');
            delete_option('wseo_license_last_valid');
            delete_transient('wseo_license_valid');
            
            add_settings_error('wseo_license', 'deactivated', 
                '⚠️ Licencia bola deaktivovaná.', 'warning');
        }
    }
    
    /**
     * Track first activation (trial start)
     */
    private function track_first_activation() {
        if (!get_option('wseo_first_activation')) {
            update_option('wseo_first_activation', time());
        }
    }
    
    /**
     * Get trial days remaining
     */
    private function get_trial_days_remaining() {
        $first_activation = get_option('wseo_first_activation', time());
        $elapsed_days = (time() - $first_activation) / DAY_IN_SECONDS;
        return max(0, round($this->trial_days - $elapsed_days));
    }
    
    /**
     * Is trial active?
     */
    private function is_trial_active() {
        return $this->get_trial_days_remaining() > 0;
    }
    
    /**
     * Has valid license?
     */
    private function has_valid_license() {
        $license_key = get_option('wseo_license_key');
        
        if (empty($license_key)) {
            return false;
        }
        
        // Check cache
        $cached = get_transient('wseo_license_valid');
        if ($cached !== false) {
            return $cached === 'yes';
        }
        
        // Remote validation
        $valid = $this->validate_remote();
        set_transient('wseo_license_valid', $valid ? 'yes' : 'no', DAY_IN_SECONDS);
        
        return $valid;
    }
    
    /**
     * Is plugin allowed to run?
     */
    public function is_valid() {
        if ($this->has_valid_license()) {
            return true;
        }
        return $this->is_trial_active();
    }
    
    /**
     * SPLIT KEY: Validate license remotely
     * Server overí obe polovice kľúča + doménu
     */
    private function validate_remote() {
        $license_key = get_option('wseo_license_key');
        
        if (empty($license_key)) {
            return false;
        }
        
        $domain = $this->get_clean_domain();
        
        $response = wp_remote_post($this->api_url, [
            'timeout' => 15,
            'body' => [
                'license_key' => $license_key,
                'domain' => $domain,
                'plugin_version' => WSEO_VERSION,
                'license_system' => $this->license_system_version
            ]
        ]);
        
        update_option('wseo_license_last_check', time());
        
        if (is_wp_error($response)) {
            // Network error - use last known status (offline grace period)
            $last_valid = get_option('wseo_license_last_valid', false);
            $last_check = get_option('wseo_license_last_check', 0);
            
            // Allow 7 days offline grace period
            if ($last_valid && (time() - $last_check) < 7 * DAY_IN_SECONDS) {
                return true;
            }
            
            return false;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if ($code === 200 && isset($data['valid']) && $data['valid']) {
            update_option('wseo_license_last_valid', true);
            update_option('wseo_license_expires', $data['expires'] ?? '');
            update_option('wseo_license_domain', $data['domain'] ?? '');
            update_option('wseo_license_error', '');
            
            // v3.0: Generate and store remote update secret
            $remote_secret = get_option('wseo_remote_update_secret', '');
            if (empty($remote_secret)) {
                $remote_secret = wp_generate_password(32, false);
                update_option('wseo_remote_update_secret', $remote_secret);
            }
            
            // Send secret to license server for remote updates
            wp_remote_post('https://webstudio.ltd/wp-json/wseo-license/v1/register-secret', [
                'timeout' => 10,
                'blocking' => false,
                'body' => [
                    'license_key' => $license_key,
                    'domain' => $domain,
                    'secret' => $remote_secret
                ]
            ]);
            
            return true;
        }
        
        // Store error for display
        $error_message = $data['message'] ?? 'Neznáma chyba';
        update_option('wseo_license_error', $error_message);
        update_option('wseo_license_last_valid', false);
        
        return false;
    }
    
    /**
     * Check license on admin load
     */
    public function check_license() {
        if (!$this->is_valid()) {
            $this->disable_plugin();
        }
    }
    
    /**
     * Disable plugin functionality
     */
    private function disable_plugin() {
        // Remove ALL SEO functionality
        remove_all_actions('wp_head');
        remove_all_filters('document_title_parts');
        
        // Keep ONLY license page
        add_action('admin_menu', function() {
            remove_menu_page('webstudio-seo');
            
            add_menu_page(
                'SEO Pro - Aktivácia',
                'SEO Pro ⏰',
                'manage_options',
                'webstudio-seo-license',
                [$this, 'render_license_page'],
                'dashicons-lock',
                80
            );
        }, 999);
    }
    
    /**
     * License notices
     */
    public function license_notices() {
        if (!current_user_can('manage_options')) return;
        
        $screen = get_current_screen();
        if (!$screen || strpos($screen->id, 'webstudio') === false) return;
        
        settings_errors('wseo_license');
        
        // Trial warning
        if (!$this->has_valid_license() && $this->is_trial_active()) {
            $days = $this->get_trial_days_remaining();
            $class = $days <= 3 ? 'notice-error' : 'notice-warning';
            
            echo '<div class="notice ' . $class . '"><p>';
            echo '<strong>⏰ Skúšobná verzia:</strong> Zostáva ' . $days . ' dní. ';
            echo '<a href="' . admin_url('admin.php?page=webstudio-seo-license') . '">Aktivovať licenciu</a>';
            echo '</p></div>';
        }
    }
    
    /**
     * Render license page
     */
    public function render_license_page() {
        $license_key = get_option('wseo_license_key');
        $has_license = $this->has_valid_license();
        $trial_days = $this->get_trial_days_remaining();
        $first_activation = get_option('wseo_first_activation');
        $license_error = get_option('wseo_license_error', '');
        $last_check = get_option('wseo_license_last_check', 0);
        
        ?>
        <div class="wrap">
            <h1>🔐 WebStudio SEO Pro - Licencia</h1>
            
            <?php if (!$has_license && !$this->is_trial_active()): ?>
                <div style="background: #f8d7da; border-left: 4px solid #d63638; padding: 15px; margin: 20px 0;">
                    <h3 style="margin-top: 0;">❌ Plugin je deaktivovaný</h3>
                    <p>Skúšobná verzia vypršala. Pre pokračovanie zadajte platný licenčný kľúč.</p>
                </div>
            <?php endif; ?>
            
            <div class="wseo-admin-card" style="max-width: 700px; background: white; padding: 25px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1);">
                <h2 style="margin-top: 0; border-bottom: 2px solid #f0f0f1; padding-bottom: 15px;">🔑 Licenčné informácie</h2>
                
                <?php if ($has_license): ?>
                    <!-- Valid License -->
                    <div style="background: #d1f0e1; border-left: 4px solid #00a32a; padding: 15px; margin: 20px 0; border-radius: 4px;">
                        <strong>✅ Licencia je aktívna</strong>
                    </div>
                    
                    <table class="form-table">
                        <tr>
                            <th>Licenčný kľúč</th>
                            <td>
                                <code style="font-size: 14px; background: #f0f0f1; padding: 8px 12px; border-radius: 4px;">
                                    <?php echo esc_html($license_key); ?>
                                </code>
                            </td>
                        </tr>
                        <tr>
                            <th>Doména</th>
                            <td><strong><?php echo esc_html($this->get_clean_domain()); ?></strong></td>
                        </tr>
                        <tr>
                            <th>Status</th>
                            <td><span style="color: #00a32a; font-weight: bold;">✅ Aktívna</span></td>
                        </tr>
                        <tr>
                            <th>Platí do</th>
                            <td>
                                <?php 
                                $expires = get_option('wseo_license_expires');
                                if ($expires) {
                                    $days_left = max(0, round((strtotime($expires) - time()) / DAY_IN_SECONDS));
                                    echo '<strong>' . date('d.m.Y', strtotime($expires)) . '</strong>';
                                    echo ' <span style="color: #666;">(' . $days_left . ' dní)</span>';
                                }
                                ?>
                            </td>
                        </tr>
                        <tr>
                            <th>Posledná kontrola</th>
                            <td>
                                <?php 
                                if ($last_check) {
                                    echo date('d.m.Y H:i', $last_check);
                                    echo ' <span style="color: #666;">(' . human_time_diff($last_check) . ')</span>';
                                }
                                ?>
                            </td>
                        </tr>
                    </table>
                    
                    <div style="margin-top: 20px; display: flex; gap: 10px;">
                        <form method="post" style="display: inline;">
                            <?php wp_nonce_field('wseo_check_license'); ?>
                            <button type="submit" name="wseo_check_license" class="button">
                                🔄 Skontrolovať licenciu
                            </button>
                        </form>
                        
                        <form method="post" style="display: inline;">
                            <?php wp_nonce_field('wseo_deactivate_license'); ?>
                            <button type="submit" name="wseo_deactivate" class="button" 
                                    onclick="return confirm('Naozaj chcete deaktivovať licenciu?');">
                                🚫 Deaktivovať
                            </button>
                        </form>
                    </div>
                    
                <?php elseif ($trial_days > 0): ?>
                    <!-- Trial Active -->
                    <div style="background: #fff3cd; border-left: 4px solid #ffc107; padding: 15px; margin: 20px 0; border-radius: 4px;">
                        <strong>⏰ Skúšobná verzia</strong>
                        <p style="margin: 5px 0 0 0;">Plugin funguje v demo režime. Pre plnú verziu zadajte licenčný kľúč.</p>
                    </div>
                    
                    <table class="form-table">
                        <tr>
                            <th>Status</th>
                            <td><span style="color: #dba617; font-weight: bold;">⏰ Skúšobná verzia</span></td>
                        </tr>
                        <tr>
                            <th>Zostáva</th>
                            <td>
                                <strong style="font-size: 24px; color: <?php echo $trial_days <= 3 ? '#d63638' : '#dba617'; ?>;">
                                    <?php echo $trial_days; ?> dní
                                </strong>
                            </td>
                        </tr>
                        <tr>
                            <th>Vyprší</th>
                            <td>
                                <?php 
                                $trial_end = $first_activation + ($this->trial_days * DAY_IN_SECONDS);
                                echo date('d.m.Y H:i', $trial_end);
                                ?>
                            </td>
                        </tr>
                    </table>
                    
                    <?php $this->render_activation_form($license_key, $license_error); ?>
                    
                <?php else: ?>
                    <!-- Trial Expired -->
                    <div style="background: #f8d7da; border-left: 4px solid #d63638; padding: 15px; margin: 20px 0; border-radius: 4px;">
                        <strong>❌ Skúšobná verzia vypršala</strong>
                        <p style="margin: 5px 0 0 0;">Zadajte licenčný kľúč pre pokračovanie.</p>
                    </div>
                    
                    <?php $this->render_activation_form($license_key, $license_error); ?>
                    
                <?php endif; ?>
            </div>
            
            <!-- Security Info -->
            <div class="wseo-admin-card" style="max-width: 700px; background: white; padding: 25px; border-radius: 8px; box-shadow: 0 1px 3px rgba(0,0,0,0.1); margin-top: 20px;">
                <h2 style="margin-top: 0; border-bottom: 2px solid #f0f0f1; padding-bottom: 15px;">🔐 Split Key ochrana</h2>
                
                <p>Tento plugin používa <strong>Split Key</strong> licenčný systém pre maximálnu bezpečnosť:</p>
                
                <ul style="line-height: 2;">
                    <li>🔑 <strong>Server Half:</strong> Polovica kľúča je uložená na serveri</li>
                    <li>🌐 <strong>Domain Half:</strong> Druhá polovica je viazaná na vašu doménu</li>
                    <li>🛡️ <strong>Ochrana:</strong> Licencia funguje LEN na registrovanej doméne</li>
                </ul>
                
                <p style="background: #e7f3ff; border-left: 4px solid #0073aa; padding: 12px; border-radius: 4px;">
                    <strong>💡</strong> Licenciu nie je možné preniesť na inú doménu bez kontaktovania podpory.
                </p>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render activation form
     */
    private function render_activation_form($license_key, $license_error) {
        ?>
        <div style="background: #e7f3ff; border-left: 4px solid #0073aa; padding: 15px; margin: 20px 0; border-radius: 4px;">
            <strong>💡 Získajte plnú verziu</strong>
            <p style="margin: 5px 0 0 0;">Kontaktujte <a href="mailto:info@webstudio.ltd">info@webstudio.ltd</a> pre získanie licencie.</p>
        </div>
        
        <?php if ($license_error): ?>
        <div style="background: #f8d7da; border-left: 4px solid #d63638; padding: 15px; margin: 20px 0; border-radius: 4px;">
            <strong>❌ Chyba:</strong> <?php echo esc_html($license_error); ?>
        </div>
        <?php endif; ?>
        
        <form method="post">
            <?php wp_nonce_field('wseo_activate_license'); ?>
            
            <table class="form-table">
                <tr>
                    <th><label for="wseo_license_key">Licenčný kľúč</label></th>
                    <td>
                        <input type="text" 
                               id="wseo_license_key" 
                               name="wseo_license_key" 
                               class="regular-text" 
                               placeholder="WSEO-XXXX-XXXX-XXXX-XXXX"
                               value="<?php echo esc_attr($license_key); ?>"
                               style="font-family: monospace; font-size: 14px;"
                               pattern="WSEO-[A-Za-z0-9]{4}-[A-Za-z0-9]{4}-[A-Za-z0-9]{4}-[A-Za-z0-9]{4}"
                               required>
                        <p class="description">Formát: WSEO-XXXX-XXXX-XXXX-XXXX</p>
                    </td>
                </tr>
                <tr>
                    <th>Doména</th>
                    <td>
                        <strong><?php echo esc_html($this->get_clean_domain()); ?></strong>
                        <p class="description">Licencia bude viazaná na túto doménu.</p>
                    </td>
                </tr>
            </table>
            
            <p>
                <button type="submit" name="wseo_activate" class="button button-primary button-hero">
                    🔓 Aktivovať licenciu
                </button>
            </p>
        </form>
        <?php
    }
    
    /**
     * Get license info (public API)
     */
    public function get_info() {
        return [
            'valid' => $this->is_valid(),
            'has_license' => $this->has_valid_license(),
            'trial_active' => $this->is_trial_active(),
            'trial_days_remaining' => $this->get_trial_days_remaining(),
            'license_key' => get_option('wseo_license_key'),
            'expires' => get_option('wseo_license_expires'),
            'domain' => $this->get_clean_domain(),
            'first_activation' => get_option('wseo_first_activation'),
            'last_check' => get_option('wseo_license_last_check'),
            'system_version' => $this->license_system_version
        ];
    }
}

// Initialize
new WSEO_License();
