<?php
/**
 * Auto-Update Client
 * 
 * @package WebStudio_SEO_Pro
 * @version 2.8.9-fix
 * 
 * CHANGELOG:
 * - FIX: Update notice persists after successful update
 * - FIX: Added cleanup on upgrader_process_complete hook
 * - FIX: Clear transients and options when versions match
 */

if (!defined('ABSPATH')) exit;

class WSEO_Auto_Update {
    
    // v2.9.16: Changed to REST API endpoint
    private $update_url = 'https://webstudio.ltd/wp-json/wseo-updates/v1/check';
    private $plugin_slug = 'webstudio-seo-pro/webstudio-seo-pro.php';
    private $plugin_basename;
    
    public function __construct() {
        $this->plugin_basename = plugin_basename(WSEO_PLUGIN_DIR . 'webstudio-seo-pro.php');
        
        // Hook into WordPress update system
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_updates']);
        add_filter('plugins_api', [$this, 'plugin_info'], 10, 3);
        
        // Check updates twice daily
        add_action('init', [$this, 'schedule_update_check']);
        add_action('wseo_check_updates_cron', [$this, 'force_check_updates']);
        
        // Admin notice for updates
        add_action('admin_notices', [$this, 'update_notice']);
        
        // AJAX handler for dismiss
        add_action('wp_ajax_wseo_dismiss_update', [$this, 'ajax_dismiss_update']);
        
        // ===== FIX: Clean up after successful update =====
        add_action('upgrader_process_complete', [$this, 'after_update_cleanup'], 10, 2);
        
        // ===== FIX: Check and clean on every admin load =====
        add_action('admin_init', [$this, 'maybe_cleanup_update_notice']);
        
        // v2.9.16: REST API endpoint for push updates from license server
        add_action('rest_api_init', [$this, 'register_push_update_endpoint']);
    }
    
    /**
     * v2.9.16: Register REST API endpoint for push updates
     */
    public function register_push_update_endpoint() {
        register_rest_route('wseo-pro/v1', '/force-update', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_push_update'],
            'permission_callback' => '__return_true'
        ]);
        
        // v3.0: Remote auto-update endpoint - actually installs the update!
        register_rest_route('wseo-pro/v1', '/remote-update', [
            'methods' => 'POST',
            'callback' => [$this, 'handle_remote_update'],
            'permission_callback' => '__return_true'
        ]);
    }
    
    /**
     * v3.0: Handle REMOTE UPDATE - Actually install the update!
     * This is called from license server to update plugin without manual intervention
     */
    public function handle_remote_update($request) {
        $license_key = $request->get_param('license_key');
        $secret = $request->get_param('secret');
        $version = $request->get_param('version');
        $download_url = $request->get_param('download_url');
        
        // Verify license key
        $stored_license = get_option('wseo_license_key', '');
        if (empty($license_key) || $license_key !== $stored_license) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'LICENSE_MISMATCH',
                'message' => 'License key mismatch'
            ], 403);
        }
        
        // Verify secret
        $stored_secret = get_option('wseo_remote_update_secret', '');
        if (empty($secret) || $secret !== $stored_secret) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'INVALID_SECRET',
                'message' => 'Invalid or missing secret'
            ], 403);
        }
        
        // Check if already on this version or newer
        if (version_compare(WSEO_VERSION, $version, '>=')) {
            return new WP_REST_Response([
                'success' => true,
                'message' => 'Already up to date',
                'current_version' => WSEO_VERSION
            ], 200);
        }
        
        // Validate download URL
        if (empty($download_url) || !filter_var($download_url, FILTER_VALIDATE_URL)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'INVALID_URL',
                'message' => 'Invalid download URL'
            ], 400);
        }
        
        // Include WordPress upgrade functions
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/misc.php';
        
        // Silent skin - no output
        $skin = new WP_Ajax_Upgrader_Skin();
        $upgrader = new Plugin_Upgrader($skin);
        
        // Get plugin file path
        $plugin_file = 'webstudio-seo-pro/webstudio-seo-pro.php';
        
        // Perform the upgrade
        $result = $upgrader->upgrade($plugin_file, [
            'clear_update_cache' => true,
            'overwrite_package' => true,
            // Custom package URL
        ]);
        
        // Since upgrade() doesn't support custom URL directly, we need different approach
        // Download and install manually
        
        // 1. Download the ZIP
        $temp_file = download_url($download_url, 300);
        
        if (is_wp_error($temp_file)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'DOWNLOAD_FAILED',
                'message' => $temp_file->get_error_message()
            ], 500);
        }
        
        // 2. Unzip to plugins directory
        $plugin_dir = WP_PLUGIN_DIR;
        
        // Remove existing plugin folder
        $existing_folder = $plugin_dir . '/webstudio-seo-pro';
        if (is_dir($existing_folder)) {
            // Deactivate first
            deactivate_plugins($plugin_file, true);
            
            // Remove folder
            global $wp_filesystem;
            WP_Filesystem();
            $wp_filesystem->delete($existing_folder, true);
        }
        
        // 3. Unzip new version
        $unzip_result = unzip_file($temp_file, $plugin_dir);
        
        // Clean up temp file
        @unlink($temp_file);
        
        if (is_wp_error($unzip_result)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'UNZIP_FAILED',
                'message' => $unzip_result->get_error_message()
            ], 500);
        }
        
        // 4. Reactivate plugin
        $activate_result = activate_plugin($plugin_file);
        
        if (is_wp_error($activate_result)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'ACTIVATION_FAILED',
                'message' => $activate_result->get_error_message()
            ], 500);
        }
        
        // 5. Clear caches
        delete_transient('wseo_remote_version');
        delete_option('wseo_update_available');
        delete_site_transient('update_plugins');
        
        // Get new version (re-read from file)
        $plugin_data = get_plugin_data($plugin_dir . '/webstudio-seo-pro/webstudio-seo-pro.php');
        $new_version = $plugin_data['Version'] ?? $version;
        
        return new WP_REST_Response([
            'success' => true,
            'message' => 'Plugin updated successfully',
            'old_version' => WSEO_VERSION,
            'new_version' => $new_version,
            'updated_at' => current_time('mysql')
        ], 200);
    }
    
    /**
     * v2.9.16: Handle push update request from license server
     */
    public function handle_push_update($request) {
        $license_key = $request->get_param('license_key');
        $secret = $request->get_param('secret');
        $version = $request->get_param('version');
        $download_url = $request->get_param('download_url');
        
        // Verify this site's license key matches
        $stored_license = get_option('wseo_license_key', '');
        
        if (empty($license_key) || $license_key !== $stored_license) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'License key mismatch'
            ], 403);
        }
        
        // Simple secret check - must not be empty
        if (empty($secret)) {
            return new WP_REST_Response([
                'success' => false,
                'message' => 'Missing secret'
            ], 403);
        }
        
        // Clear update cache to force re-check
        delete_transient('wseo_remote_version');
        delete_site_transient('update_plugins');
        
        // Store update info to trigger notice immediately
        update_option('wseo_update_available', [
            'version' => $version,
            'download_url' => $download_url,
            'changelog' => ['Push update zo servera'],
            'push_update' => true,
            'pushed_at' => current_time('mysql')
        ]);
        
        return new WP_REST_Response([
            'success' => true,
            'message' => 'Update notification received',
            'current_version' => WSEO_VERSION,
            'new_version' => $version
        ], 200);
    }
    
    /**
     * Schedule update checks
     */
    public function schedule_update_check() {
        if (!wp_next_scheduled('wseo_check_updates_cron')) {
            wp_schedule_event(time(), 'twicedaily', 'wseo_check_updates_cron');
        }
    }
    
    /**
     * Check for updates
     */
    public function check_for_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }
        
        // Get remote version
        $remote = $this->get_remote_version();
        
        if (!$remote) {
            return $transient;
        }
        
        // Compare versions
        if (version_compare(WSEO_VERSION, $remote['version'], '<')) {
            // New version available!
            $plugin_data = [
                'slug' => 'webstudio-seo-pro',
                'new_version' => $remote['version'],
                'url' => 'https://webstudio.ltd',
                'package' => $remote['download_url'],
                'tested' => $remote['tested'] ?? '6.4',
                'requires_php' => $remote['requires_php'] ?? '7.4',
                'requires' => $remote['requires'] ?? '5.8'
            ];
            
            $transient->response[$this->plugin_basename] = (object) $plugin_data;
            
            // Store update info for notices
            update_option('wseo_update_available', $remote);
        } else {
            // ===== FIX: No update available - ALWAYS clean up =====
            $this->cleanup_update_data();
            
            // Remove from WordPress update response if present
            if (isset($transient->response[$this->plugin_basename])) {
                unset($transient->response[$this->plugin_basename]);
            }
        }
        
        return $transient;
    }
    
    /**
     * Get remote version info
     */
    private function get_remote_version() {
        // Check cache first (1h instead of 12h for faster response)
        $cached = get_transient('wseo_remote_version');
        
        if ($cached !== false) {
            return $cached;
        }
        
        // Fetch from server
        $response = wp_remote_get($this->update_url, [
            'timeout' => 10,
            'headers' => [
                'User-Agent' => 'WebStudio-SEO-Pro/' . WSEO_VERSION . '; ' . home_url()
            ]
        ]);
        
        if (is_wp_error($response)) {
            return false;
        }
        
        $data = json_decode(wp_remote_retrieve_body($response), true);
        
        if (!$data || !isset($data['version'])) {
            return false;
        }
        
        // Cache for 1 hour (reduced from 12h)
        set_transient('wseo_remote_version', $data, HOUR_IN_SECONDS);
        
        return $data;
    }
    
    /**
     * Provide plugin info for update screen
     */
    public function plugin_info($false, $action, $args) {
        if ($action !== 'plugin_information') {
            return $false;
        }
        
        if (!isset($args->slug) || $args->slug !== 'webstudio-seo-pro') {
            return $false;
        }
        
        $remote = $this->get_remote_version();
        
        if (!$remote) {
            return $false;
        }
        
        // Build changelog HTML
        $changelog_html = '<h4>Zmeny vo verzii ' . esc_html($remote['version']) . ':</h4>';
        $changelog_html .= '<ul>';
        
        if (!empty($remote['changelog'])) {
            foreach ($remote['changelog'] as $item) {
                $changelog_html .= '<li>' . esc_html($item) . '</li>';
            }
        }
        
        $changelog_html .= '</ul>';
        
        $info = (object) [
            'name' => 'WebStudio SEO Pro',
            'slug' => 'webstudio-seo-pro',
            'version' => $remote['version'],
            'author' => '<a href="https://webstudio.ltd">Webstudio Ltd</a>',
            'homepage' => 'https://webstudio.ltd',
            'download_link' => $remote['download_url'],
            'requires' => $remote['requires'] ?? '5.8',
            'tested' => $remote['tested'] ?? '6.4',
            'requires_php' => $remote['requires_php'] ?? '7.4',
            'last_updated' => $remote['last_updated'] ?? date('Y-m-d'),
            'sections' => [
                'description' => 'Kompletné SEO riešenie pre WordPress s 200+ funkciami, license systémom a auto-update.',
                'changelog' => $changelog_html
            ],
            'banners' => [],
            'icons' => []
        ];
        
        return $info;
    }
    
    /**
     * Force check for updates
     */
    public function force_check_updates() {
        delete_transient('wseo_remote_version');
        delete_site_transient('update_plugins');
        
        // Trigger update check
        wp_update_plugins();
    }
    
    /**
     * ===== FIX: Cleanup after successful plugin update =====
     */
    public function after_update_cleanup($upgrader_object, $options) {
        // Check if our plugin was updated
        if ($options['action'] === 'update' && $options['type'] === 'plugin') {
            $plugins = $options['plugins'] ?? [];
            
            // Check both possible plugin paths
            $our_plugin = false;
            foreach ($plugins as $plugin) {
                if (strpos($plugin, 'webstudio-seo-pro') !== false) {
                    $our_plugin = true;
                    break;
                }
            }
            
            if ($our_plugin) {
                // Clear ALL update-related data
                $this->cleanup_update_data();
                
                // Also clear WordPress plugin transients
                delete_site_transient('update_plugins');
                
                // Log for debugging
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    error_log('WSEO: Update cleanup performed after upgrader_process_complete');
                }
            }
        }
    }
    
    /**
     * ===== FIX: Check on every admin page load if we need cleanup =====
     */
    public function maybe_cleanup_update_notice() {
        $update = get_option('wseo_update_available');
        
        if (!$update) {
            return;
        }
        
        // If stored version equals or is less than current version, clean up!
        if (isset($update['version']) && version_compare(WSEO_VERSION, $update['version'], '>=')) {
            $this->cleanup_update_data();
            
            // Log for debugging
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('WSEO: Update notice cleaned - current version ' . WSEO_VERSION . ' >= ' . $update['version']);
            }
        }
    }
    
    /**
     * ===== NEW: Centralized cleanup function =====
     */
    private function cleanup_update_data() {
        // Delete update available option
        delete_option('wseo_update_available');
        
        // Delete remote version cache
        delete_transient('wseo_remote_version');
        
        // Delete user dismissals for all versions (clean slate)
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->usermeta} WHERE meta_key LIKE 'wseo_dismissed_update_%'");
    }
    
    /**
     * Admin notice for available updates
     */
    public function update_notice() {
        if (!current_user_can('update_plugins')) {
            return;
        }
        
        $screen = get_current_screen();
        
        // Don't show on plugins page (WordPress shows own notice)
        if ($screen && $screen->id === 'plugins') {
            return;
        }
        
        $update = get_option('wseo_update_available');
        
        if (!$update || !isset($update['version'])) {
            return;
        }
        
        // ===== FIX: Double-check that update is actually needed =====
        if (version_compare(WSEO_VERSION, $update['version'], '>=')) {
            // We already have this version or newer - clean up and don't show
            $this->cleanup_update_data();
            return;
        }
        
        // Check if user dismissed this version
        $dismissed = get_user_meta(get_current_user_id(), 'wseo_dismissed_update_' . $update['version'], true);
        
        if ($dismissed) {
            return;
        }
        
        ?>
        <div class="notice notice-info is-dismissible" data-wseo-update-notice="<?php echo esc_attr($update['version']); ?>">
            <p>
                <strong>🎉 WebStudio SEO Pro v<?php echo esc_html($update['version']); ?> je dostupná!</strong>
                <small style="color: #666;">(Aktuálna verzia: <?php echo esc_html(WSEO_VERSION); ?>)</small>
            </p>
            <?php 
            // v3.2: Ensure changelog is array before using array_slice
            $changelog = isset($update['changelog']) ? $update['changelog'] : array();
            if (is_string($changelog)) {
                $changelog = array_filter(array_map('trim', explode("\n", $changelog)));
            }
            if (!is_array($changelog)) {
                $changelog = array();
            }
            if (!empty($changelog)): 
            ?>
                <p style="margin-left: 20px;">
                    <strong>Novinky:</strong><br>
                    <?php foreach (array_slice($changelog, 0, 3) as $item): ?>
                        • <?php echo esc_html($item); ?><br>
                    <?php endforeach; ?>
                </p>
            <?php endif; ?>
            <p>
                <a href="<?php echo admin_url('plugins.php'); ?>" class="button button-primary">
                    Aktualizovať teraz
                </a>
                <a href="#" class="button wseo-dismiss-update">
                    Neskôr
                </a>
            </p>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('.wseo-dismiss-update').on('click', function(e) {
                e.preventDefault();
                
                var $notice = $(this).closest('.notice');
                var version = $notice.data('wseo-update-notice');
                
                $.post(ajaxurl, {
                    action: 'wseo_dismiss_update',
                    version: version,
                    nonce: '<?php echo wp_create_nonce('wseo_dismiss_update'); ?>'
                }, function() {
                    $notice.fadeOut();
                });
            });
            
            // Also handle WordPress native dismiss button
            $('[data-wseo-update-notice]').on('click', '.notice-dismiss', function() {
                var $notice = $(this).closest('.notice');
                var version = $notice.data('wseo-update-notice');
                
                $.post(ajaxurl, {
                    action: 'wseo_dismiss_update',
                    version: version,
                    nonce: '<?php echo wp_create_nonce('wseo_dismiss_update'); ?>'
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX: Dismiss update notice
     */
    public function ajax_dismiss_update() {
        check_ajax_referer('wseo_dismiss_update', 'nonce');
        
        $version = isset($_POST['version']) ? sanitize_text_field($_POST['version']) : '';
        
        if (!empty($version)) {
            update_user_meta(get_current_user_id(), 'wseo_dismissed_update_' . $version, true);
        }
        
        wp_send_json_success();
    }
}

// Initialize
new WSEO_Auto_Update();
