/**
 * Intelligent Page Prefetching
 * WebStudio SEO Pro v2.6.3
 * © 2026 Karol Kameniczki - Webstudio.ltd
 * 
 * IMPORTANT: This script is COMPLETELY DISABLED for logged-in users
 * to prevent any conflicts with WordPress admin bar.
 */
(function() {
    'use strict';
    
    // ========================================
    // SAFETY CHECKS - Exit immediately if:
    // 1. User is logged in (admin bar)
    // 2. Page builder is active
    // 3. Admin/preview mode
    // ========================================
    
    /**
     * CRITICAL: Check if script should be disabled
     * Must return true to STOP script execution
     */
    function shouldDisable() {
        // Check URL first (fastest)
        var url = window.location.href;
        if (url.indexOf('/wp-admin') !== -1 ||
            url.indexOf('wp-login') !== -1 ||
            url.indexOf('elementor-preview') !== -1 ||
            url.indexOf('tb-preview') !== -1 ||
            url.indexOf('fl_builder') !== -1 ||
            url.indexOf('et_fb=1') !== -1 ||
            url.indexOf('bricks=') !== -1 ||
            url.indexOf('ct_builder') !== -1 ||
            url.indexOf('vc_editable') !== -1 ||
            url.indexOf('preview=true') !== -1 ||
            url.indexOf('preview_id=') !== -1) {
            return true;
        }
        
        // Check for admin bar element (most reliable for logged-in)
        if (document.getElementById('wpadminbar')) {
            return true;
        }
        
        // Check html classes
        var html = document.documentElement;
        if (html.classList.contains('wp-toolbar')) {
            return true;
        }
        
        // Check body classes (if body exists)
        var body = document.body;
        if (body) {
            if (body.classList.contains('logged-in') ||
                body.classList.contains('admin-bar') ||
                body.classList.contains('wp-admin') ||
                body.classList.contains('elementor-editor-active') ||
                body.classList.contains('themify_builder_active') ||
                body.classList.contains('tb-editor-active') ||
                body.classList.contains('fl-builder-edit') ||
                body.classList.contains('et-fb') ||
                body.classList.contains('bricks-is-frontend') ||
                body.classList.contains('oxygen-builder-body') ||
                body.classList.contains('vc_editor')) {
                return true;
            }
        }
        
        // Check for builder globals
        if (window.elementor || window.elementorFrontend ||
            window.ThemifyBuilder || window.themifyBuilder ||
            window.FLBuilder || window.ET_Builder ||
            window.bricksData || window.oxygen ||
            window.vc || (window.wp && window.wp.customize)) {
            return true;
        }
        
        return false;
    }
    
    // ========================================
    // MAIN LOGIC - Only runs for visitors
    // ========================================
    
    var config = window.wseoConfig || {
        prefetchEnabled: true,
        prefetchStrategy: 'hover',
        prefetchHoverDelay: 250,  // Increased from 65ms - more conservative
        prefetchExclude: [],
        prefetchMaxConcurrent: 1, // Decreased from 3 - less aggressive
        prefetchMaxPages: 5       // Decreased from 10 - prevent overload
    };
    
    var prefetchedUrls = {};
    var currentPrefetches = 0;
    var maxPrefetches = config.prefetchMaxConcurrent || 1;
    var maxPages = config.prefetchMaxPages || 5;
    var prefetchCount = 0;
    
    /**
     * Check if URL is valid for prefetch
     */
    function isValidUrl(url) {
        if (!url) return false;
        
        try {
            var urlObj = new URL(url, window.location.origin);
            
            // Must be same origin
            if (urlObj.origin !== window.location.origin) return false;
            
            // Not current page
            if (urlObj.pathname === window.location.pathname) return false;
            
            // Not admin URLs
            if (urlObj.pathname.indexOf('/wp-admin') !== -1) return false;
            if (urlObj.pathname.indexOf('/wp-login') !== -1) return false;
            
            // Not already prefetched
            if (prefetchedUrls[url]) return false;
            
            // Check exclude patterns
            for (var i = 0; i < config.prefetchExclude.length; i++) {
                try {
                    var regex = new RegExp(config.prefetchExclude[i]);
                    if (regex.test(url)) return false;
                } catch (e) {}
            }
            
            return true;
        } catch (e) {
            return false;
        }
    }
    
    /**
     * Check if connection is fast enough for prefetch
     */
    function isFastConnection() {
        // Check Network Information API
        if ('connection' in navigator) {
            var conn = navigator.connection || navigator.mozConnection || navigator.webkitConnection;
            if (conn) {
                // Don't prefetch on slow connections
                if (conn.saveData) return false; // Data saver mode
                if (conn.effectiveType === 'slow-2g') return false;
                if (conn.effectiveType === '2g') return false;
                
                // Only prefetch on fast connections
                if (conn.effectiveType === '4g') return true;
                if (conn.effectiveType === '3g') return true;
            }
        }
        
        // If unknown, be conservative - only prefetch on good connections
        return true;
    }
    
    /**
     * Prefetch a URL using link element
     */
    function prefetchUrl(url) {
        if (!isValidUrl(url)) return;
        if (currentPrefetches >= maxPrefetches) return;
        if (prefetchCount >= maxPages) return;
        
        // IMPORTANT: Don't prefetch on slow connections
        if (!isFastConnection()) return;
        
        prefetchedUrls[url] = true;
        currentPrefetches++;
        prefetchCount++;
        
        var link = document.createElement('link');
        link.rel = 'prefetch';
        link.href = url;
        link.as = 'document';
        
        link.onload = function() {
            currentPrefetches--;
        };
        link.onerror = function() {
            currentPrefetches--;
            delete prefetchedUrls[url];
            prefetchCount--;
        };
        
        document.head.appendChild(link);
    }
    
    /**
     * Hover prefetch strategy
     */
    function initHoverPrefetch() {
        var hoverTimer;
        var delay = config.prefetchHoverDelay || 250; // Increased from 65ms
        var currentLink = null;
        
        // Use event delegation for better performance
        document.addEventListener('mouseover', function(e) {
            var link = e.target.closest ? e.target.closest('a[href]') : null;
            if (!link || link === currentLink) return;
            
            currentLink = link;
            clearTimeout(hoverTimer);
            hoverTimer = setTimeout(function() {
                prefetchUrl(link.href);
            }, delay);
        }, { passive: true });
        
        document.addEventListener('mouseout', function(e) {
            var link = e.target.closest ? e.target.closest('a[href]') : null;
            if (link === currentLink) {
                clearTimeout(hoverTimer);
                currentLink = null;
            }
        }, { passive: true });
    }
    
    /**
     * Viewport prefetch strategy
     */
    function initViewportPrefetch() {
        if (!('IntersectionObserver' in window)) return;
        
        var observer = new IntersectionObserver(function(entries) {
            entries.forEach(function(entry) {
                if (entry.isIntersecting) {
                    prefetchUrl(entry.target.href);
                    observer.unobserve(entry.target);
                }
            });
        }, { rootMargin: '50px' });
        
        var links = document.querySelectorAll('a[href^="/"], a[href^="' + window.location.origin + '"]');
        links.forEach(function(link) {
            if (isValidUrl(link.href)) {
                observer.observe(link);
            }
        });
    }
    
    /**
     * Initialize prefetching
     */
    function init() {
        // CRITICAL: Final safety check
        if (shouldDisable()) {
            return;
        }
        
        if (!config.prefetchEnabled) {
            return;
        }
        
        // Use simple hover strategy (safest)
        if (config.prefetchStrategy === 'viewport') {
            initViewportPrefetch();
        } else {
            initHoverPrefetch();
        }
    }
    
    // ========================================
    // START - Wait for DOM to be ready
    // ========================================
    
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        // Small delay to ensure body classes are set
        setTimeout(init, 10);
    }
    
})();
