/**
 * Add custom cron schedules
 */
public function add_cron_schedules($schedules) {
    $schedules['weekly'] = array(
        'interval' => 604800, // 7 days
        'display' => 'Raz týždenne'
    );
    $schedules['monthly'] = array(
        'interval' => 2635200, // 30 days
        'display' => 'Raz mesačne'
    );
    return $schedules;
}

/**
 * AJAX: Cleanup selected items
 */
public function ajax_maintenance_cleanup() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    $items = isset($_POST['items']) ? (array)$_POST['items'] : array();
    
    if (empty($items)) {
        wp_send_json_error('Neboli vybrané žiadne položky.');
    }
    
    global $wpdb;
    $deleted = array();
    $total_deleted = 0;
    
    foreach ($items as $item) {
        $count = 0;
        
        switch ($item) {
            case 'revisions':
                $count = $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_type = 'revision'");
                $deleted['Revízie'] = $count;
                break;
                
            case 'autodrafts':
                $count = $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_status = 'auto-draft'");
                $deleted['Auto-drafts'] = $count;
                break;
                
            case 'trash':
                $count = $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_status = 'trash'");
                $deleted['Kôš'] = $count;
                break;
                
            case 'spam_comments':
                $count = $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = 'spam'");
                $deleted['Spam komentáre'] = $count;
                break;
                
            case 'trashed_comments':
                $count = $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = 'trash'");
                $deleted['Vymazané komentáre'] = $count;
                break;
                
            case 'expired_transients':
                $count = $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%' AND option_value < UNIX_TIMESTAMP()");
                $count += $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_%' AND option_name NOT LIKE '_transient_timeout_%' AND option_name NOT IN (SELECT REPLACE(option_name, '_timeout', '') FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%')");
                $deleted['Expirované transienty'] = $count;
                break;
        }
        
        $total_deleted += $count;
    }
    
    // Log activity
    $this->log_maintenance_activity('Čistenie databázy', 'Vymazané položky: ' . implode(', ', array_map(function($k, $v) {
        return "$k ($v)";
    }, array_keys($deleted), $deleted)));
    
    // Update last cleanup time
    update_option('wseo_last_cleanup_time', time());
    
    $message = '<strong>Vymazané položky:</strong><br>';
    foreach ($deleted as $type => $count) {
        $message .= "• $type: " . number_format($count) . " záznamov<br>";
    }
    $message .= "<br><strong>Celkom:</strong> " . number_format($total_deleted) . " záznamov";
    
    wp_send_json_success(array('message' => $message, 'deleted' => $deleted));
}

/**
 * AJAX: Optimize database tables
 */
public function ajax_maintenance_optimize() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    global $wpdb;
    
    // Get all tables
    $tables = $wpdb->get_results("SHOW TABLES", ARRAY_N);
    
    $optimized = 0;
    $total_overhead = 0;
    
    foreach ($tables as $table) {
        $table_name = $table[0];
        
        // Get table status
        $status = $wpdb->get_row("SHOW TABLE STATUS LIKE '$table_name'");
        if ($status && isset($status->Data_free)) {
            $total_overhead += $status->Data_free;
        }
        
        // Optimize table
        $wpdb->query("OPTIMIZE TABLE `$table_name`");
        $optimized++;
    }
    
    // Log activity
    $this->log_maintenance_activity('Optimalizácia databázy', "Optimalizované tabulky: $optimized, Uvoľnený priestor: " . size_format($total_overhead));
    
    $message = "<strong>Optimalizované tabulky:</strong> $optimized<br>";
    $message .= "<strong>Uvoľnený overhead:</strong> " . size_format($total_overhead);
    
    wp_send_json_success(array('message' => $message));
}

/**
 * AJAX: Clean orphaned metadata
 */
public function ajax_maintenance_orphaned() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    $type = isset($_POST['type']) ? sanitize_text_field($_POST['type']) : '';
    
    global $wpdb;
    $deleted = 0;
    
    switch ($type) {
        case 'postmeta':
            $deleted = $wpdb->query("
                DELETE pm FROM {$wpdb->postmeta} pm
                LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID
                WHERE p.ID IS NULL
            ");
            $type_label = 'Post meta';
            break;
            
        case 'commentmeta':
            $deleted = $wpdb->query("
                DELETE cm FROM {$wpdb->commentmeta} cm
                LEFT JOIN {$wpdb->comments} c ON cm.comment_id = c.comment_ID
                WHERE c.comment_ID IS NULL
            ");
            $type_label = 'Comment meta';
            break;
            
        case 'termmeta':
            $deleted = $wpdb->query("
                DELETE tm FROM {$wpdb->termmeta} tm
                LEFT JOIN {$wpdb->terms} t ON tm.term_id = t.term_id
                WHERE t.term_id IS NULL
            ");
            $type_label = 'Term meta';
            break;
            
        default:
            wp_send_json_error('Neplatný typ metadát.');
            return;
    }
    
    // Log activity
    $this->log_maintenance_activity('Čistenie orphaned metadata', "$type_label: $deleted záznamov");
    
    $message = "<strong>$type_label:</strong> Vymazané " . number_format($deleted) . " orphaned záznamov";
    
    wp_send_json_success(array('message' => $message));
}

/**
 * AJAX: Scan autoload options
 */
public function ajax_maintenance_scan_autoload() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    global $wpdb;
    
    // Get top 20 largest autoload options
    $results = $wpdb->get_results("
        SELECT option_name, LENGTH(option_value) as size
        FROM {$wpdb->options}
        WHERE autoload = 'yes'
        ORDER BY size DESC
        LIMIT 20
    ");
    
    if (empty($results)) {
        wp_send_json_error('Nepodarilo sa načítať autoload options.');
    }
    
    $html = '<div class="wseo-result-box">';
    $html .= '<h3 style="margin-top: 0;">Top 20 najväčších autoload options:</h3>';
    $html .= '<table class="wp-list-table widefat fixed striped">';
    $html .= '<thead><tr><th>Option Name</th><th>Veľkosť</th><th>Akcia</th></tr></thead>';
    $html .= '<tbody>';
    
    foreach ($results as $row) {
        $size = size_format($row->size, 2);
        $option_name = esc_html($row->option_name);
        $html .= '<tr>';
        $html .= '<td><code>' . $option_name . '</code></td>';
        $html .= '<td>' . $size . '</td>';
        $html .= '<td><button class="button button-small wseo-disable-autoload" data-option="' . esc_attr($option_name) . '">Vypnúť autoload</button></td>';
        $html .= '</tr>';
    }
    
    $html .= '</tbody></table>';
    $html .= '<p style="margin-top: 15px;"><strong>Poznámka:</strong> Vypnutie autoload pre systémové options môže spôsobiť problémy. Buďte opatrní!</p>';
    $html .= '</div>';
    
    $html .= '<script>
    jQuery(document).ready(function($) {
        $(".wseo-disable-autoload").on("click", function() {
            var option = $(this).data("option");
            if (!confirm("Ste si istí, že chcete vypnúť autoload pre: " + option + "?")) {
                return;
            }
            
            $.ajax({
                url: ajaxurl,
                type: "POST",
                data: {
                    action: "wseo_maintenance_disable_autoload",
                    option: option,
                    nonce: "' . wp_create_nonce('wseo_maintenance') . '"
                },
                success: function(response) {
                    if (response.success) {
                        alert("✓ Autoload vypnutý!");
                        location.reload();
                    } else {
                        alert("✗ Chyba: " + response.data);
                    }
                }
            });
        });
    });
    </script>';
    
    wp_send_json_success(array('html' => $html));
}

/**
 * Helper: Disable autoload for option
 */
public function ajax_maintenance_disable_autoload() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    $option = isset($_POST['option']) ? sanitize_text_field($_POST['option']) : '';
    
    if (empty($option)) {
        wp_send_json_error('Neplatný option name.');
    }
    
    global $wpdb;
    
    $result = $wpdb->update(
        $wpdb->options,
        array('autoload' => 'no'),
        array('option_name' => $option),
        array('%s'),
        array('%s')
    );
    
    if ($result !== false) {
        $this->log_maintenance_activity('Autoload optimization', "Vypnutý autoload pre: $option");
        wp_send_json_success();
    } else {
        wp_send_json_error('Nepodarilo sa aktualizovať option.');
    }
}

/**
 * AJAX: Save scheduled cleanup settings
 */
public function ajax_maintenance_save_scheduled() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    $settings = array(
        'enabled' => isset($_POST['enabled']) && $_POST['enabled'] === 'true',
        'frequency' => isset($_POST['frequency']) ? sanitize_text_field($_POST['frequency']) : 'weekly',
        'items' => isset($_POST['items']) ? (array)$_POST['items'] : array(),
        'email_report' => isset($_POST['email_report']) && $_POST['email_report'] === 'true'
    );
    
    update_option('wseo_scheduled_cleanup', $settings);
    
    // Clear existing schedule
    wp_clear_scheduled_hook('wseo_scheduled_maintenance');
    
    // Schedule new event if enabled
    if ($settings['enabled']) {
        $recurrence = 'daily';
        if ($settings['frequency'] === 'weekly') {
            $recurrence = 'weekly';
        } elseif ($settings['frequency'] === 'monthly') {
            $recurrence = 'monthly';
        }
        
        wp_schedule_event(time(), $recurrence, 'wseo_scheduled_maintenance');
    }
    
    $this->log_maintenance_activity('Scheduled cleanup nastavenia', 'Aktualizované nastavenia automatického čistenia');
    
    wp_send_json_success();
}

/**
 * Run scheduled maintenance
 */
public function run_scheduled_maintenance() {
    $settings = get_option('wseo_scheduled_cleanup', array());
    
    if (empty($settings['enabled']) || empty($settings['items'])) {
        return;
    }
    
    global $wpdb;
    $deleted = array();
    
    foreach ($settings['items'] as $item) {
        $count = 0;
        
        switch ($item) {
            case 'revisions':
                $count = $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_type = 'revision'");
                $deleted['Revízie'] = $count;
                break;
                
            case 'autodrafts':
                $count = $wpdb->query("DELETE FROM {$wpdb->posts} WHERE post_status = 'auto-draft'");
                $deleted['Auto-drafts'] = $count;
                break;
                
            case 'expired_transients':
                $count = $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%' AND option_value < UNIX_TIMESTAMP()");
                $deleted['Expirované transienty'] = $count;
                break;
                
            case 'spam_comments':
                $count = $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_approved = 'spam'");
                $deleted['Spam komentáre'] = $count;
                break;
        }
    }
    
    // Log activity
    $this->log_maintenance_activity('Automatické čistenie', 'Vymazané položky: ' . implode(', ', array_map(function($k, $v) {
        return "$k ($v)";
    }, array_keys($deleted), $deleted)));
    
    // Send email report if enabled
    if (!empty($settings['email_report'])) {
        $admin_email = get_option('admin_email');
        $site_name = get_bloginfo('name');
        
        $message = "Automatické čistenie databázy pre $site_name\n\n";
        $message .= "Dátum: " . date('d.m.Y H:i') . "\n\n";
        $message .= "Vymazané položky:\n";
        
        foreach ($deleted as $type => $count) {
            $message .= "• $type: " . number_format($count) . " záznamov\n";
        }
        
        wp_mail($admin_email, "[{$site_name}] Report z automatického čistenia databázy", $message);
    }
    
    update_option('wseo_last_cleanup_time', time());
}

/**
 * Log maintenance activity
 */
private function log_maintenance_activity($action, $details) {
    $log = get_option('wseo_maintenance_log', array());
    
    $log[] = array(
        'time' => date('d.m.Y H:i:s'),
        'action' => $action,
        'details' => $details
    );
    
    // Keep only last 100 entries
    if (count($log) > 100) {
        $log = array_slice($log, -100);
    }
    
    update_option('wseo_maintenance_log', $log);
}

/**
 * AJAX: Clear maintenance log
 */
public function ajax_maintenance_clear_log() {
    check_ajax_referer('wseo_maintenance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie vykonať túto akciu.');
    }
    
    update_option('wseo_maintenance_log', array());
    wp_send_json_success();
}
