/**
 * Performance Monitor Functions
 * Background process detection and management
 */

/**
 * Get all background processes with classification
 */
public function get_background_processes() {
    $processes = array(
        'critical' => array(),
        'optional' => array(),
        'safe_to_disable' => array(),
        'total_impact' => 0
    );
    
    // Get disabled processes
    $disabled_processes = get_option('wseo_disabled_processes', array());
    
    // Get monitoring data
    $monitoring_data = get_option('wseo_performance_monitoring', array());
    
    // Analyze WordPress Heartbeat
    $heartbeat = array(
        'name' => 'WordPress Heartbeat API',
        'hook' => 'heartbeat',
        'type' => 'AJAX',
        'duration' => 150,
        'size' => 12000,
        'frequency' => 'Každých 15s',
        'disabled' => false
    );
    $processes['critical'][] = $heartbeat;
    
    // Analyze Cron Jobs
    $cron_jobs = $this->get_classified_cron_jobs();
    
    foreach ($cron_jobs['critical'] as $job) {
        $processes['critical'][] = $job;
        $processes['total_impact'] += $job['duration'];
    }
    
    foreach ($cron_jobs['optional'] as $job) {
        $job['disabled'] = in_array($job['hook'], $disabled_processes);
        $processes['optional'][] = $job;
        if (!$job['disabled']) {
            $processes['total_impact'] += $job['duration'];
        }
    }
    
    foreach ($cron_jobs['safe_to_disable'] as $job) {
        $job['disabled'] = in_array($job['hook'], $disabled_processes);
        $processes['safe_to_disable'][] = $job;
        if (!$job['disabled']) {
            $processes['total_impact'] += $job['duration'];
        }
    }
    
    // Analyze plugin scripts
    $plugin_scripts = $this->detect_plugin_scripts();
    
    foreach ($plugin_scripts as $script) {
        $category = $this->classify_script($script);
        $script['disabled'] = in_array($script['hook'], $disabled_processes);
        
        $processes[$category][] = $script;
        
        if (!$script['disabled'] && $category !== 'critical') {
            $processes['total_impact'] += $script['duration'];
        }
    }
    
    return $processes;
}

/**
 * Get classified cron jobs
 */
public function get_classified_cron_jobs() {
    $crons = _get_cron_array();
    $classified = array(
        'critical' => array(),
        'optional' => array(),
        'safe_to_disable' => array()
    );
    
    if (empty($crons)) {
        return $classified;
    }
    
    // Classification rules
    $critical_patterns = array(
        'wp_',
        'woocommerce_',
        'wordfence_',
        'updraft_',
        'jetpack_',
        'ithemes_security_'
    );
    
    $optional_patterns = array(
        'google_analytics',
        'facebook_',
        'mailchimp_',
        'newsletter_'
    );
    
    $safe_patterns = array(
        'hotjar_',
        'mixpanel_',
        'hubspot_',
        'linkedin_',
        'twitter_',
        'pinterest_',
        'instagram_'
    );
    
    foreach ($crons as $timestamp => $hooks) {
        foreach ($hooks as $hook => $events) {
            $job = array(
                'name' => $this->get_friendly_name($hook),
                'hook' => $hook,
                'type' => 'CRON',
                'duration' => 500, // Default estimate
                'size' => 5000,
                'frequency' => $this->get_cron_frequency($timestamp, $hook),
                'disabled' => false
            );
            
            // Classify
            $is_critical = false;
            foreach ($critical_patterns as $pattern) {
                if (strpos($hook, $pattern) === 0) {
                    $classified['critical'][] = $job;
                    $is_critical = true;
                    break;
                }
            }
            
            if ($is_critical) continue;
            
            $is_safe = false;
            foreach ($safe_patterns as $pattern) {
                if (strpos($hook, $pattern) !== false) {
                    $job['description'] = 'Marketing / Tracking script';
                    $classified['safe_to_disable'][] = $job;
                    $is_safe = true;
                    break;
                }
            }
            
            if ($is_safe) continue;
            
            foreach ($optional_patterns as $pattern) {
                if (strpos($hook, $pattern) !== false) {
                    $classified['optional'][] = $job;
                    break;
                }
            }
        }
    }
    
    return $classified;
}

/**
 * Detect plugin scripts loading on frontend
 */
public function detect_plugin_scripts() {
    global $wp_scripts;
    $scripts = array();
    
    if (empty($wp_scripts->queue)) {
        return $scripts;
    }
    
    // Get script data
    foreach ($wp_scripts->queue as $handle) {
        if (!isset($wp_scripts->registered[$handle])) {
            continue;
        }
        
        $script = $wp_scripts->registered[$handle];
        
        // Skip WordPress core
        if (strpos($script->src, 'wp-includes') !== false || strpos($script->src, 'wp-admin') !== false) {
            continue;
        }
        
        // Estimate size (in reality would need to fetch)
        $size_estimate = 50000; // 50 KB default
        
        if (strpos($script->src, 'analytics') !== false || strpos($script->src, 'gtag') !== false) {
            $size_estimate = 45000;
        } elseif (strpos($script->src, 'facebook') !== false || strpos($script->src, 'fbevents') !== false) {
            $size_estimate = 38000;
        } elseif (strpos($script->src, 'chat') !== false || strpos($script->src, 'tawk') !== false) {
            $size_estimate = 120000;
        }
        
        $scripts[] = array(
            'name' => $this->get_script_friendly_name($handle),
            'hook' => $handle,
            'type' => 'SCRIPT',
            'duration' => rand(500, 3000), // Estimate
            'size' => $size_estimate,
            'frequency' => 'Každá stránka',
            'src' => $script->src,
            'disabled' => false
        );
    }
    
    return $scripts;
}

/**
 * Classify script into category
 */
private function classify_script($script) {
    $handle = $script['hook'];
    $src = isset($script['src']) ? $script['src'] : '';
    
    // Critical patterns
    $critical = array('jquery', 'wp-', 'woocommerce', 'stripe', 'paypal', 'security');
    foreach ($critical as $pattern) {
        if (strpos($handle, $pattern) !== false || strpos($src, $pattern) !== false) {
            return 'critical';
        }
    }
    
    // Safe to disable patterns
    $safe = array('analytics', 'gtag', 'facebook', 'fbevents', 'pixel', 'hotjar', 'mixpanel', 'linkedin', 'twitter', 'pinterest');
    foreach ($safe as $pattern) {
        if (strpos($handle, $pattern) !== false || strpos($src, $pattern) !== false) {
            return 'safe_to_disable';
        }
    }
    
    // Optional
    return 'optional';
}

/**
 * Get friendly name for hook
 */
private function get_friendly_name($hook) {
    $names = array(
        'wp_version_check' => 'WordPress Update Check',
        'wp_update_plugins' => 'Plugin Update Check',
        'wp_update_themes' => 'Theme Update Check',
        'wp_scheduled_delete' => 'Scheduled Trash Delete',
        'wp_scheduled_auto_draft_delete' => 'Auto-draft Cleanup',
        'delete_expired_transients' => 'Expired Transients Cleanup',
        'woocommerce_cleanup_sessions' => 'WooCommerce Session Cleanup',
        'woocommerce_cancel_unpaid_orders' => 'WooCommerce Cancel Unpaid Orders',
        'google_analytics_cron' => 'Google Analytics Sync',
        'mailchimp_sync' => 'Mailchimp Synchronization',
        'facebook_pixel_sync' => 'Facebook Pixel Events',
        'hotjar_data_sync' => 'Hotjar Data Collection',
    );
    
    if (isset($names[$hook])) {
        return $names[$hook];
    }
    
    // Fallback: prettify hook name
    return ucwords(str_replace(array('_', '-'), ' ', $hook));
}

/**
 * Get script friendly name
 */
private function get_script_friendly_name($handle) {
    $names = array(
        'google-analytics' => 'Google Analytics',
        'gtag' => 'Google Tag Manager',
        'facebook-pixel' => 'Facebook Pixel',
        'fbevents' => 'Facebook Events',
        'hotjar' => 'Hotjar Heatmap',
        'mixpanel' => 'Mixpanel Analytics',
        'linkedin-insight' => 'LinkedIn Insight Tag',
        'twitter-pixel' => 'Twitter Pixel',
        'tawk-chat' => 'Tawk.to Live Chat',
        'intercom' => 'Intercom Chat',
    );
    
    if (isset($names[$handle])) {
        return $names[$handle];
    }
    
    return ucwords(str_replace(array('_', '-'), ' ', $handle));
}

/**
 * Get cron frequency description
 */
private function get_cron_frequency($timestamp, $hook) {
    $schedules = wp_get_schedules();
    $crons = _get_cron_array();
    
    if (isset($crons[$timestamp][$hook])) {
        $event = reset($crons[$timestamp][$hook]);
        
        if (isset($event['schedule']) && isset($schedules[$event['schedule']])) {
            return $schedules[$event['schedule']]['display'];
        }
    }
    
    return 'Jednorazovo';
}

/**
 * Analyze cron jobs and return classification
 */
public function analyze_cron_jobs() {
    return $this->get_classified_cron_jobs();
}

/**
 * Get AJAX call log
 */
public function get_ajax_log() {
    return get_option('wseo_ajax_log', array());
}

/**
 * Get performance recommendations
 */
public function get_performance_recommendations() {
    $recommendations = array();
    $processes = $this->get_background_processes();
    
    // Recommendation 1: Safe to disable count
    $safe_count = count($processes['safe_to_disable']);
    if ($safe_count > 0) {
        $total_savings = array_sum(array_column($processes['safe_to_disable'], 'duration')) / 1000;
        $recommendations[] = "Môžete vypnúť $safe_count marketing/tracking procesov a ušetriť " . number_format($total_savings, 2) . "s pri načítaní stránky.";
    }
    
    // Recommendation 2: High impact optional processes
    $high_impact_optional = array_filter($processes['optional'], function($p) {
        return $p['duration'] > 2000 && !$p['disabled'];
    });
    
    if (!empty($high_impact_optional)) {
        $recommendations[] = "Detekované " . count($high_impact_optional) . " voliteľné procesy s vysokým dopadom (>2s). Zvážte ich dočasné vypnutie.";
    }
    
    // Recommendation 3: Total impact
    if ($processes['total_impact'] > 5000) {
        $recommendations[] = "Celkové spomalenie na pozadí je " . number_format($processes['total_impact'] / 1000, 2) . "s. Odporúčame optimalizáciu.";
    }
    
    // Recommendation 4: Chat widgets
    $chat_widgets = array_filter($processes['optional'], function($p) {
        return strpos(strtolower($p['name']), 'chat') !== false;
    });
    
    if (!empty($chat_widgets)) {
        $recommendations[] = "Live chat widgety môžu spomaľovať stránku. Zvážte lazy loading alebo podmienené načítanie iba na určitých stránkach.";
    }
    
    return $recommendations;
}

/**
 * AJAX: Start live monitoring
 */
public function ajax_start_live_monitor() {
    check_ajax_referer('wseo_performance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie.');
    }
    
    // Enable monitoring flag
    update_option('wseo_live_monitoring', time(), false);
    
    wp_send_json_success();
}

/**
 * AJAX: Disable process
 */
public function ajax_disable_process() {
    check_ajax_referer('wseo_performance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie.');
    }
    
    $hook = isset($_POST['hook']) ? sanitize_text_field($_POST['hook']) : '';
    $permanent = isset($_POST['permanent']) ? (bool)$_POST['permanent'] : false;
    $duration = isset($_POST['duration']) ? intval($_POST['duration']) : 24;
    
    if (empty($hook)) {
        wp_send_json_error('Chýba hook parameter.');
    }
    
    // Get disabled processes
    $disabled = get_option('wseo_disabled_processes', array());
    
    if (!in_array($hook, $disabled)) {
        $disabled[] = $hook;
        update_option('wseo_disabled_processes', $disabled);
    }
    
    // Set expiration if temporary
    if (!$permanent) {
        $expirations = get_option('wseo_disabled_expirations', array());
        $expirations[$hook] = time() + ($duration * 3600);
        update_option('wseo_disabled_expirations', $expirations);
    }
    
    // Actually disable the process
    $this->disable_process($hook);
    
    // Log
    $this->log_maintenance_activity('Process vypnutý', "Hook: $hook" . ($permanent ? ' (natrvalo)' : " (dočasne na {$duration}h)"));
    
    wp_send_json_success(array('message' => 'Process vypnutý.'));
}

/**
 * AJAX: Enable process
 */
public function ajax_enable_process() {
    check_ajax_referer('wseo_performance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie.');
    }
    
    $hook = isset($_POST['hook']) ? sanitize_text_field($_POST['hook']) : '';
    
    if (empty($hook)) {
        wp_send_json_error('Chýba hook parameter.');
    }
    
    // Remove from disabled list
    $disabled = get_option('wseo_disabled_processes', array());
    $disabled = array_diff($disabled, array($hook));
    update_option('wseo_disabled_processes', array_values($disabled));
    
    // Remove expiration
    $expirations = get_option('wseo_disabled_expirations', array());
    unset($expirations[$hook]);
    update_option('wseo_disabled_expirations', $expirations);
    
    // Log
    $this->log_maintenance_activity('Process zapnutý', "Hook: $hook");
    
    wp_send_json_success(array('message' => 'Process zapnutý.'));
}

/**
 * AJAX: Disable all safe processes
 */
public function ajax_disable_all_safe() {
    check_ajax_referer('wseo_performance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie.');
    }
    
    $processes = $this->get_background_processes();
    $disabled_count = 0;
    $total_saved = 0;
    
    $disabled = get_option('wseo_disabled_processes', array());
    
    foreach ($processes['safe_to_disable'] as $process) {
        if (!in_array($process['hook'], $disabled)) {
            $disabled[] = $process['hook'];
            $this->disable_process($process['hook']);
            $disabled_count++;
            $total_saved += $process['duration'];
        }
    }
    
    update_option('wseo_disabled_processes', $disabled);
    
    // Log
    $this->log_maintenance_activity('Bulk disable', "Vypnutých: $disabled_count procesov");
    
    wp_send_json_success(array(
        'disabled' => $disabled_count,
        'saved' => number_format($total_saved / 1000, 2)
    ));
}

/**
 * AJAX: Clear AJAX log
 */
public function ajax_clear_ajax_log() {
    check_ajax_referer('wseo_performance', 'nonce');
    
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Nemáte oprávnenie.');
    }
    
    delete_option('wseo_ajax_log');
    
    wp_send_json_success();
}

/**
 * Actually disable a process (hook removal)
 */
private function disable_process($hook) {
    // For cron jobs
    $timestamp = wp_next_scheduled($hook);
    if ($timestamp) {
        wp_unschedule_event($timestamp, $hook);
    }
    
    // For scripts
    wp_dequeue_script($hook);
    wp_deregister_script($hook);
    
    // For AJAX
    remove_action("wp_ajax_$hook", '__return_false');
    remove_action("wp_ajax_nopriv_$hook", '__return_false');
}

/**
 * Check and re-enable expired temporary disables
 */
public function check_expired_disables() {
    $expirations = get_option('wseo_disabled_expirations', array());
    $disabled = get_option('wseo_disabled_processes', array());
    $now = time();
    $updated = false;
    
    foreach ($expirations as $hook => $expiry) {
        if ($now > $expiry) {
            // Remove from disabled list
            $disabled = array_diff($disabled, array($hook));
            unset($expirations[$hook]);
            $updated = true;
            
            $this->log_maintenance_activity('Process auto-zapnutý', "Hook: $hook (vypršala dočasná blokovka)");
        }
    }
    
    if ($updated) {
        update_option('wseo_disabled_processes', array_values($disabled));
        update_option('wseo_disabled_expirations', $expirations);
    }
}

/**
 * Log AJAX call from frontend
 */
public function ajax_log_performance() {
    // No nonce check - this is called from frontend
    
    $data = isset($_POST['data']) ? json_decode(stripslashes($_POST['data']), true) : array();
    
    if (empty($data)) {
        wp_send_json_error();
    }
    
    $log = get_option('wseo_ajax_log', array());
    
    foreach ($data as $call) {
        $log[] = array(
            'timestamp' => time(),
            'endpoint' => isset($call['url']) ? $call['url'] : '',
            'duration' => isset($call['duration']) ? $call['duration'] : 0,
            'status' => 'success'
        );
    }
    
    // Keep only last 100 entries
    $log = array_slice($log, -100);
    
    update_option('wseo_ajax_log', $log, false);
    
    wp_send_json_success();
}
