<?php
/**
 * Performance Monitor Template
 * Real-time background process detection and management
 */

if (!defined('ABSPATH')) exit;

// Get performance data
$background_processes = $this->get_background_processes();
$cron_jobs = $this->analyze_cron_jobs();
$ajax_log = $this->get_ajax_log();
$recommendations = $this->get_performance_recommendations();
$smart_recommendations = $this->get_smart_recommendations();

?>

<div class="wrap wseo-admin-wrap">
    <h1>⚡ Performance Monitor</h1>
    
    <?php if (isset($_GET['heartbeat_saved'])): ?>
        <div class="notice notice-success is-dismissible">
            <p><strong>✓ Heartbeat nastavenia uložené!</strong> Zmeny sa prejavia okamžite.</p>
        </div>
    <?php endif; ?>
    
    <div style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
        <h2 style="color: white; margin: 0 0 10px;">Real-time Background Process Detection</h2>
        <p style="margin: 0; font-size: 15px; opacity: 0.95;">
            Identifikuje všetky procesy bežiace na pozadí, ktoré môžu spomaľovať vašu stránku.
            Color-coded klasifikácia: 🔴 Kritické | 🟡 Voliteľné | 🟢 Bezpečné vypnúť
        </p>
    </div>
    
    <!-- Live Status Dashboard -->
    <div class="wseo-admin-card" style="background: #f8f9fa;">
        <div style="display: grid; grid-template-columns: repeat(4, 1fr); gap: 15px; margin-bottom: 20px;">
            <div style="background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <div style="font-size: 36px; font-weight: bold; color: #dc3545;" id="wseo-critical-count">
                    <?php echo count($background_processes['critical']); ?>
                </div>
                <div style="font-size: 13px; color: #6c757d; margin-top: 5px;">🔴 Kritické procesy</div>
            </div>
            <div style="background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <div style="font-size: 36px; font-weight: bold; color: #ffc107;" id="wseo-optional-count">
                    <?php echo count($background_processes['optional']); ?>
                </div>
                <div style="font-size: 13px; color: #6c757d; margin-top: 5px;">🟡 Voliteľné procesy</div>
            </div>
            <div style="background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <div style="font-size: 36px; font-weight: bold; color: #28a745;" id="wseo-safe-count">
                    <?php echo count($background_processes['safe_to_disable']); ?>
                </div>
                <div style="font-size: 13px; color: #6c757d; margin-top: 5px;">🟢 Bezpečné vypnúť</div>
            </div>
            <div style="background: white; padding: 20px; border-radius: 8px; text-align: center; box-shadow: 0 2px 4px rgba(0,0,0,0.1);">
                <div style="font-size: 36px; font-weight: bold; color: #2196f3;" id="wseo-total-impact">
                    <?php echo number_format($background_processes['total_impact'] / 1000, 2); ?>s
                </div>
                <div style="font-size: 13px; color: #6c757d; margin-top: 5px;">⏱️ Celkové spomalenie</div>
            </div>
        </div>
        
        <div style="text-align: center; margin-bottom: 20px;">
            <button type="button" class="button button-primary button-large" id="wseo-start-live-monitor">
                <span class="dashicons dashicons-visibility" style="margin-top: 3px;"></span>
                Spustiť Live Monitoring (30s)
            </button>
            <button type="button" class="button button-secondary" id="wseo-refresh-data">
                <span class="dashicons dashicons-update" style="margin-top: 3px;"></span>
                Obnoviť dáta
            </button>
        </div>
        
        <div id="wseo-live-status" style="display: none; background: #fff; padding: 15px; border-radius: 4px; border-left: 4px solid #2196f3;">
            <strong>🔴 LIVE:</strong> <span id="wseo-live-message">Monitorujem pozadie...</span>
            <div class="spinner is-active" style="float: right; margin: 0;"></div>
        </div>
    </div>
    
    <!-- Recommendations -->
    <?php if (!empty($recommendations)): ?>
    <div class="wseo-admin-card" style="background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%); color: white; border: none;">
        <h2 style="color: white; margin-top: 0;">💡 Odporúčania na zrýchlenie</h2>
        <div style="background: rgba(255,255,255,0.95); padding: 20px; border-radius: 8px; color: #333;">
            <?php foreach ($recommendations as $rec): ?>
                <div style="margin-bottom: 20px; padding-bottom: 20px; border-bottom: 1px solid #eee;">
                    <p style="margin: 0 0 10px 0; font-size: 15px; line-height: 1.6;">
                        <strong><?php echo esc_html($rec['message']); ?></strong>
                    </p>
                    
                    <?php if (!empty($rec['process_list'])): ?>
                        <div style="background: #f8f9fa; padding: 12px; border-radius: 5px; margin: 10px 0;">
                            <strong>📋 Zoznam procesov:</strong>
                            <ul style="margin: 8px 0 0 20px; padding: 0;">
                                <?php foreach ($rec['process_list'] as $proc_name): ?>
                                    <li style="margin: 5px 0;"><?php echo esc_html($proc_name); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($rec['processes'])): ?>
                        <div style="margin-top: 10px;">
                            <details style="cursor: pointer;">
                                <summary style="font-weight: bold; color: #2271b1; padding: 8px 0;">
                                    🔍 Zobraziť detaily (<?php echo count($rec['processes']); ?> procesov)
                                </summary>
                                <div style="margin-top: 10px; padding: 10px; background: #f8f9fa; border-radius: 5px;">
                                    <?php foreach ($rec['processes'] as $process): ?>
                                        <div style="display: flex; justify-content: space-between; align-items: center; padding: 8px; background: white; margin-bottom: 5px; border-radius: 3px;">
                                            <div>
                                                <strong><?php echo esc_html($process['name']); ?></strong>
                                                <span style="color: #666; font-size: 12px;">
                                                    (<?php echo esc_html($process['source']); ?>)
                                                </span>
                                            </div>
                                            <div style="display: flex; gap: 10px; align-items: center;">
                                                <span style="color: #d63638; font-weight: bold;">
                                                    <?php echo number_format($process['duration'] / 1000, 2); ?>s
                                                </span>
                                                <?php if (!$process['disabled']): ?>
                                                    <button class="button button-small wseo-disable-process" 
                                                            data-hook="<?php echo esc_attr($process['hook']); ?>"
                                                            style="padding: 2px 10px; height: auto;">
                                                        Vypnúť
                                                    </button>
                                                <?php else: ?>
                                                    <span style="color: #00a32a; font-size: 12px;">✓ Vypnuté</span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    <?php endforeach; ?>
                                </div>
                            </details>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($rec['action_label']) && !empty($rec['action_id'])): ?>
                        <div style="margin-top: 15px;">
                            <button id="<?php echo esc_attr($rec['action_id']); ?>" 
                                    class="button button-primary" 
                                    style="background: #2271b1; border: none; box-shadow: none;">
                                <?php echo esc_html($rec['action_label']); ?>
                            </button>
                        </div>
                    <?php endif; ?>
                    
                    <?php if (!empty($rec['scroll_to'])): ?>
                        <div style="margin-top: 15px;">
                            <button class="button button-secondary wseo-scroll-to" 
                                    data-target="<?php echo esc_attr($rec['scroll_to']); ?>">
                                <?php echo esc_html($rec['action_label'] ?: 'Zobraziť'); ?>
                            </button>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Smart Recommendations -->
    <?php if (!empty($smart_recommendations)): ?>
    <div class="wseo-admin-card" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; border: none;">
        <h2 style="color: white; margin-bottom: 15px;">
            🤖 AI Odporúčania - Automatická detekcia problémov
        </h2>
        <p style="color: rgba(255,255,255,0.9); margin-bottom: 20px;">
            Analyzovali sme admin-ajax.php logy za posledných 24 hodín a identifikovali sme procesy, ktoré spomaľujú váš web.
        </p>
        
        <div style="background: rgba(255,255,255,0.95); padding: 20px; border-radius: 8px; color: #333;">
            <?php foreach ($smart_recommendations as $rec): ?>
                <div style="background: white; padding: 15px; margin-bottom: 15px; border-radius: 5px; border-left: 4px solid <?php 
                    echo $rec['severity'] === 'critical' ? '#dc3545' : 
                         ($rec['severity'] === 'high' ? '#ff6b35' : 
                         ($rec['severity'] === 'medium' ? '#ffc107' : '#28a745')); 
                ?>;">
                    <h3 style="margin: 0 0 10px 0; font-size: 16px;">
                        <?php echo esc_html($rec['action']); ?>
                        <?php if ($rec['process']): ?>
                            <span style="color: #666; font-size: 13px; font-weight: normal;">
                                (<?php echo esc_html($rec['process']['source']); ?>)
                            </span>
                        <?php endif; ?>
                    </h3>
                    
                    <p style="margin: 10px 0; line-height: 1.6;">
                        <?php echo $rec['message']; ?>
                    </p>
                    
                    <div style="display: flex; gap: 15px; align-items: center; margin-top: 10px; flex-wrap: wrap;">
                        <div style="background: #f5f5f5; padding: 8px 12px; border-radius: 4px; font-size: 13px;">
                            📊 Volania: <strong><?php echo $rec['stats']['count']; ?>x</strong>
                        </div>
                        <div style="background: #f5f5f5; padding: 8px 12px; border-radius: 4px; font-size: 13px;">
                            ⏱️ Avg čas: <strong><?php echo number_format($rec['stats']['avg_duration'] / 1000, 2); ?>s</strong>
                        </div>
                        <div style="background: #f5f5f5; padding: 8px 12px; border-radius: 4px; font-size: 13px;">
                            💾 Celkový čas: <strong><?php echo number_format($rec['stats']['total_duration'] / 1000, 2); ?>s</strong>
                        </div>
                        <div style="background: #d4edda; padding: 8px 12px; border-radius: 4px; font-size: 13px; color: #155724;">
                            ✅ Potenciálna úspora: <strong><?php echo number_format($rec['potential_savings'] / 1000, 2); ?>s</strong>
                        </div>
                    </div>
                    
                    <?php if ($rec['process']): ?>
                        <div style="margin-top: 15px; padding-top: 15px; border-top: 1px solid #eee;">
                            <button class="button button-primary wseo-apply-smart-recommendation" 
                                    data-hook="<?php echo esc_attr($rec['process']['hook']); ?>"
                                    data-delay="<?php echo esc_attr($rec['recommended_delay']); ?>"
                                    data-action="<?php echo esc_attr($rec['action']); ?>">
                                ⚡ Aplikovať odporúčanie (<?php echo ($rec['recommended_delay'] / 1000); ?>s throttle)
                            </button>
                            <span style="color: #666; font-size: 12px; margin-left: 10px;">
                                Jedným klikom optimalizujete tento proces
                            </span>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
            
            <div style="margin-top: 20px; padding: 15px; background: #e7f3ff; border-radius: 5px; border-left: 4px solid #2196F3;">
                <strong>💡 Tip:</strong> Aplikovaním všetkých odporúčaní môžete ušetriť až 
                <strong><?php 
                    $total_savings = array_sum(array_column($smart_recommendations, 'potential_savings'));
                    echo number_format($total_savings / 1000, 2); 
                ?>s</strong> pri každom načítaní stránky!
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Critical Processes -->
    <div class="wseo-admin-card">
        <h2 style="color: #dc3545;">🔴 Kritické procesy (NEDOTÝKAŤ SA)</h2>
        <p class="description">Tieto procesy sú nevyhnutné pre správne fungovanie WordPress a pluginov.</p>
        
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th style="width: 30%;">Proces</th>
                    <th style="width: 15%;">Plugin/Zdroj</th>
                    <th style="width: 12%;">Typ</th>
                    <th style="width: 13%;">Čas načítania</th>
                    <th style="width: 13%;">Veľkosť</th>
                    <th style="width: 17%;">Frekvencia</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($background_processes['critical'])): ?>
                    <tr>
                        <td colspan="6" style="text-align: center; padding: 20px;">
                            ✓ Žiadne kritické procesy nedetekované
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($background_processes['critical'] as $process): ?>
                        <tr>
                            <td>
                                <strong><?php echo esc_html($process['name']); ?></strong><br>
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($process['hook']); ?></span>
                            </td>
                            <td>
                                <strong style="color: #2271b1;">
                                    <?php echo esc_html(isset($process['source']) ? $process['source'] : 'Unknown'); ?>
                                </strong>
                            </td>
                            <td>
                                <span class="wseo-badge" style="background: #dc3545; color: white; padding: 3px 8px; border-radius: 3px; font-size: 11px;">
                                    <?php echo esc_html($process['type']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($process['duration'], 0); ?>ms</td>
                            <td><?php echo size_format($process['size']); ?></td>
                            <td><?php echo esc_html($process['frequency']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Optional Processes -->
    <div class="wseo-admin-card" id="optional-processes">
        <h2 style="color: #ffc107;">🟡 Voliteľné procesy (MÔŽU SPOMAĽOVAŤ)</h2>
        <p class="description">Tieto procesy nie sú kritické, ale odporúča sa ich nechať zapnuté pre plnú funkčnosť.</p>
        
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th style="width: 27%;">Proces</th>
                    <th style="width: 13%;">Plugin/Zdroj</th>
                    <th style="width: 10%;">Typ</th>
                    <th style="width: 10%;">Čas</th>
                    <th style="width: 10%;">Veľkosť</th>
                    <th style="width: 10%;">Impact</th>
                    <th style="width: 20%;">Akcia</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($background_processes['optional'])): ?>
                    <tr>
                        <td colspan="7" style="text-align: center; padding: 20px;">
                            ✓ Žiadne voliteľné procesy nedetekované
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($background_processes['optional'] as $process): ?>
                        <tr data-hook="<?php echo esc_attr($process['hook']); ?>">
                            <td>
                                <strong><?php echo esc_html($process['name']); ?></strong>
                                <?php if (!empty($process['adjusted_delay'])): ?>
                                    <span style="background: #00a32a; color: white; padding: 2px 6px; border-radius: 3px; font-size: 10px; margin-left: 5px;">
                                        ⏱️ <?php echo ($process['adjusted_delay'] / 1000); ?>s
                                    </span>
                                <?php endif; ?>
                                <br>
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($process['hook']); ?></span>
                            </td>
                            <td>
                                <strong style="color: #d63638;">
                                    <?php echo esc_html($process['source']); ?>
                                </strong>
                            </td>
                            <td>
                                <span class="wseo-badge" style="background: #ffc107; color: #000; padding: 3px 8px; border-radius: 3px; font-size: 11px;">
                                    <?php echo esc_html($process['type']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($process['duration'], 0); ?>ms</td>
                            <td><?php echo size_format($process['size']); ?></td>
                            <td>
                                <?php
                                $impact = $process['duration'] / 1000;
                                $impact_class = $impact > 2 ? 'high' : ($impact > 1 ? 'medium' : 'low');
                                $impact_color = $impact > 2 ? '#dc3545' : ($impact > 1 ? '#ffc107' : '#28a745');
                                ?>
                                <span style="color: <?php echo $impact_color; ?>; font-weight: bold;">
                                    <?php echo $impact > 2 ? 'Vysoký' : ($impact > 1 ? 'Stredný' : 'Nízky'); ?>
                                </span>
                            </td>
                            <td>
                                <?php if ($process['disabled']): ?>
                                    <button class="button button-secondary wseo-enable-process" data-hook="<?php echo esc_attr($process['hook']); ?>">
                                        Zapnúť
                                    </button>
                                <?php else: ?>
                                    <div style="display: flex; gap: 5px; align-items: center;">
                                        <button class="button wseo-disable-process" data-hook="<?php echo esc_attr($process['hook']); ?>">
                                            Vypnúť
                                        </button>
                                        <select class="wseo-adjust-frequency" data-hook="<?php echo esc_attr($process['hook']); ?>" style="height: 30px;">
                                            <option value="">Zmeniť čas...</option>
                                            <option value="1000">1s (menej často)</option>
                                            <option value="2000">2s (ešte menej)</option>
                                            <option value="5000">5s (veľmi zriedka)</option>
                                            <option value="10000">10s (minimum)</option>
                                        </select>
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
    
    <!-- Safe to Disable -->
    <div class="wseo-admin-card">
        <h2 style="color: #28a745;">🟢 Bezpečné vypnúť (MARKETING/TRACKING)</h2>
        <p class="description">Tieto procesy možno bezpečne vypnúť pre rýchlejšie načítanie stránky.</p>
        
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th style="width: 27%;">Proces</th>
                    <th style="width: 13%;">Plugin/Zdroj</th>
                    <th style="width: 10%;">Typ</th>
                    <th style="width: 10%;">Čas</th>
                    <th style="width: 10%;">Veľkosť</th>
                    <th style="width: 10%;">Úspora</th>
                    <th style="width: 20%;">Akcia</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($background_processes['safe_to_disable'])): ?>
                    <tr>
                        <td colspan="7" style="text-align: center; padding: 20px;">
                            ✓ Žiadne nepotrebné procesy nedetekované
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($background_processes['safe_to_disable'] as $process): ?>
                        <tr data-hook="<?php echo esc_attr($process['hook']); ?>">
                            <td>
                                <strong><?php echo esc_html($process['name']); ?></strong>
                                <?php if (!empty($process['adjusted_delay'])): ?>
                                    <span style="background: #00a32a; color: white; padding: 2px 6px; border-radius: 3px; font-size: 10px; margin-left: 5px;">
                                        ⏱️ <?php echo ($process['adjusted_delay'] / 1000); ?>s
                                    </span>
                                <?php endif; ?>
                                <br>
                                <span style="font-size: 12px; color: #666;"><?php echo esc_html($process['hook']); ?></span>
                                <?php if (!empty($process['description'])): ?>
                                    <br><em style="font-size: 11px; color: #999;"><?php echo esc_html($process['description']); ?></em>
                                <?php endif; ?>
                            </td>
                            <td>
                                <strong style="color: #00a32a;">
                                    <?php echo esc_html($process['source']); ?>
                                </strong>
                            </td>
                            <td>
                                <span class="wseo-badge" style="background: #28a745; color: white; padding: 3px 8px; border-radius: 3px; font-size: 11px;">
                                    <?php echo esc_html($process['type']); ?>
                                </span>
                            </td>
                            <td><?php echo number_format($process['duration'], 0); ?>ms</td>
                            <td><?php echo size_format($process['size']); ?></td>
                            <td>
                                <strong style="color: #28a745;">
                                    <?php echo number_format($process['duration'] / 1000, 2); ?>s
                                </strong>
                            </td>
                            <td>
                                <?php if ($process['disabled']): ?>
                                    <button class="button button-secondary wseo-enable-process" data-hook="<?php echo esc_attr($process['hook']); ?>">
                                        ✓ Vypnuté
                                    </button>
                                    <button class="button button-link-delete wseo-enable-process-permanent" data-hook="<?php echo esc_attr($process['hook']); ?>">
                                        Zapnúť
                                    </button>
                                <?php else: ?>
                                    <div style="display: flex; gap: 5px; align-items: center; flex-wrap: wrap;">
                                        <button class="button button-primary wseo-disable-process-permanent" data-hook="<?php echo esc_attr($process['hook']); ?>">
                                            Vypnúť natrvalo
                                        </button>
                                        <select class="wseo-adjust-frequency" data-hook="<?php echo esc_attr($process['hook']); ?>" style="height: 32px;">
                                            <option value="">Zmeniť čas...</option>
                                            <option value="1000">1s (menej často)</option>
                                            <option value="2000">2s (ešte menej)</option>
                                            <option value="5000">5s (veľmi zriedka)</option>
                                            <option value="10000">10s (minimum)</option>
                                        </select>
                                    </div>
                                <?php endif; ?>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
        
        <?php if (!empty($background_processes['safe_to_disable'])): ?>
            <p style="margin-top: 20px;">
                <button type="button" class="button button-primary button-large" id="wseo-disable-all-safe">
                    Vypnúť všetky bezpečné procesy
                </button>
                <span style="margin-left: 15px; color: #28a745; font-weight: bold;">
                    Potenciálna úspora: <?php echo number_format(array_sum(array_column($background_processes['safe_to_disable'], 'duration')) / 1000, 2); ?>s
                </span>
            </p>
        <?php endif; ?>
    </div>
    
    <!-- Heartbeat Control -->
    <?php
    $heartbeat_settings = get_option('wseo_heartbeat_settings', array(
        'mode' => 'default',
        'admin_interval' => 15,
        'frontend_enabled' => true,
        'editor_interval' => 15
    ));
    
    $current_load = $this->calculate_heartbeat_load($heartbeat_settings);
    
    // Get Brotli compression stats
    $compression_stats = $this->get_compression_stats();
    ?>
    
    <!-- Brotli Compression Status -->
    <div class="wseo-admin-card" style="background: linear-gradient(135deg, #a8edea 0%, #fed6e3 100%); border: none; color: #333;">
        <h2 style="color: #333; margin-top: 0;">🗜️ Brotli Compression Status</h2>
        <p style="margin: 0 0 20px; font-size: 14px; line-height: 1.6;">
            <strong>Brotli</strong> je moderná kompresná metóda až <strong>o 25% efektívnejšia</strong> než Gzip. 
            Znižuje veľkosť súborov, zrýchľuje načítanie a šetrí bandwidth.
        </p>
        
        <div style="background: rgba(255,255,255,0.9); padding: 20px; border-radius: 8px;">
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin-bottom: 20px;">
                <div style="background: white; padding: 15px; border-radius: 5px; border-left: 4px solid <?php echo $compression_stats['brotli_available'] ? '#28a745' : '#ffc107'; ?>;">
                    <h3 style="margin: 0 0 10px; font-size: 14px; color: #666;">Server Support</h3>
                    <?php if ($compression_stats['brotli_available']): ?>
                        <div style="font-size: 32px; margin: 5px 0;">✅</div>
                        <strong style="color: #28a745;">Brotli dostupný</strong>
                    <?php else: ?>
                        <div style="font-size: 32px; margin: 5px 0;">⚠️</div>
                        <strong style="color: #ffc107;">Brotli nedostupný</strong>
                    <?php endif; ?>
                </div>
                
                <div style="background: white; padding: 15px; border-radius: 5px; border-left: 4px solid <?php echo $compression_stats['brotli_enabled'] ? '#28a745' : '#dc3545'; ?>;">
                    <h3 style="margin: 0 0 10px; font-size: 14px; color: #666;">Client Support</h3>
                    <?php if ($compression_stats['brotli_enabled']): ?>
                        <div style="font-size: 32px; margin: 5px 0;">✅</div>
                        <strong style="color: #28a745;">Browser podporuje</strong>
                    <?php else: ?>
                        <div style="font-size: 32px; margin: 5px 0;">❌</div>
                        <strong style="color: #dc3545;">Browser nepodporuje</strong>
                    <?php endif; ?>
                </div>
            </div>
            
            <div style="background: #f8f9fa; padding: 15px; border-radius: 5px; margin-bottom: 15px;">
                <h3 style="margin: 0 0 10px; font-size: 14px;">📊 Technické informácie:</h3>
                <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px; font-size: 13px;">
                    <div><strong>Aktívna metóda:</strong> <span style="color: #2271b1;"><?php echo $compression_stats['default_method']; ?></span></div>
                    <div><strong>PHP verzia:</strong> <span style="color: #2271b1;"><?php echo $compression_stats['php_version']; ?></span></div>
                    <div style="grid-column: 1 / -1;"><strong>Dostupné extensions:</strong> <span style="color: #2271b1;"><?php echo implode(', ', $compression_stats['extensions']); ?></span></div>
                </div>
            </div>
            
            <?php if ($compression_stats['brotli_enabled']): ?>
                <div style="background: #d4edda; padding: 15px; border-radius: 5px; border-left: 4px solid #28a745;">
                    <strong>🎉 Brotli aktívny!</strong><br>
                    <span style="font-size: 13px;">
                        Vaše súbory sa komprimujú s Brotli, čo znamená až o 25% menšie súbory než s Gzip. 
                        Tento plugin automaticky používa najlepšiu dostupnú metódu.
                    </span>
                </div>
            <?php elseif ($compression_stats['brotli_available'] && !$compression_stats['brotli_enabled']): ?>
                <div style="background: #fff3cd; padding: 15px; border-radius: 5px; border-left: 4px solid #ffc107;">
                    <strong>⚠️ Brotli dostupný, ale browser nepodporuje</strong><br>
                    <span style="font-size: 13px;">
                        Váš server podporuje Brotli, ale váš browser ešte nie. Plugin automaticky použije Gzip fallback.
                        Modernými browsermi (Chrome, Firefox, Safari) získate Brotli automaticky.
                    </span>
                </div>
            <?php else: ?>
                <div style="background: #f8d7da; padding: 15px; border-radius: 5px; border-left: 4px solid #dc3545;">
                    <strong>📝 Ako aktivovať Brotli:</strong><br>
                    <ol style="margin: 10px 0 0 20px; padding: 0; font-size: 13px; line-height: 1.8;">
                        <li>Kontaktujte hosting support a požiadajte o aktiváciu <strong>PHP Brotli extension</strong></li>
                        <li>Alebo pridajte do <code>php.ini</code>: <code>extension=brotli</code></li>
                        <li>Po aktivácii sa Brotli automaticky zapne (bez zmeny kódu)</li>
                    </ol>
                </div>
            <?php endif; ?>
            
            <div style="margin-top: 20px; padding: 15px; background: #e7f3ff; border-radius: 5px; border-left: 4px solid #2196F3;">
                <h3 style="margin: 0 0 10px; font-size: 14px;">💡 Benefity Brotli:</h3>
                <ul style="margin: 0; padding-left: 20px; font-size: 13px; line-height: 1.8;">
                    <li>🚀 <strong>15-25% menšie súbory</strong> než Gzip</li>
                    <li>⚡ <strong>Rýchlejšie načítanie</strong> stránky</li>
                    <li>📉 <strong>Nižšia bandwidth</strong> spotreba</li>
                    <li>📈 <strong>Lepší Page Speed Score</strong></li>
                    <li>✅ <strong>100% backward compatible</strong> (automatický Gzip fallback)</li>
                </ul>
            </div>
        </div>
    </div>
    
    <div class="wseo-admin-card" style="background: linear-gradient(135deg, #fa709a 0%, #fee140 100%); border: none; color: #333;">
        <h2 style="color: #333; margin-top: 0;">💓 WordPress Heartbeat Control</h2>
        <p style="margin: 0 0 20px; font-size: 14px; line-height: 1.6;">
            WordPress Heartbeat API vytvára AJAX requesty každých <strong>15-60 sekúnd</strong> medzi browserom a serverom.
            Používa sa na auto-save, post locking a notifikácie. <strong>Vysoká server load!</strong>
        </p>
        
        <div style="background: white; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
            <div style="display: grid; grid-template-columns: repeat(3, 1fr); gap: 15px; margin-bottom: 20px;">
                <div style="text-align: center; padding: 15px; background: #f8f9fa; border-radius: 6px;">
                    <div style="font-size: 28px; font-weight: bold; color: #dc3545;">
                        <?php echo $current_load['admin_per_hour']; ?>
                    </div>
                    <div style="font-size: 12px; color: #6c757d; margin-top: 5px;">req/hod v admin (per user)</div>
                </div>
                <div style="text-align: center; padding: 15px; background: #f8f9fa; border-radius: 6px;">
                    <div style="font-size: 28px; font-weight: bold; color: <?php echo (isset($heartbeat_settings['frontend_enabled']) && $heartbeat_settings['frontend_enabled']) ? '#ffc107' : '#28a745'; ?>;">
                        <?php echo isset($current_load['frontend_per_hour']) ? $current_load['frontend_per_hour'] : 0; ?>
                    </div>
                    <div style="font-size: 12px; color: #6c757d; margin-top: 5px;">req/hod na frontende (per visitor)</div>
                </div>
                <div style="text-align: center; padding: 15px; background: #f8f9fa; border-radius: 6px;">
                    <div style="font-size: 28px; font-weight: bold; color: #2196f3;">
                        <?php echo $current_load['total_saved']; ?>%
                    </div>
                    <div style="font-size: 12px; color: #6c757d; margin-top: 5px;">úspora vs. default</div>
                </div>
            </div>
            
            <form method="post" action="" id="wseo-heartbeat-form">
                <?php wp_nonce_field('wseo_heartbeat_settings', 'wseo_heartbeat_nonce'); ?>
                
                <div style="margin-bottom: 20px;">
                    <h3 style="margin-bottom: 15px; font-size: 16px;">Vyber režim:</h3>
                    
                    <label style="display: block; padding: 15px; background: <?php echo $heartbeat_settings['mode'] === 'default' ? '#e3f2fd' : '#f8f9fa'; ?>; border: 2px solid <?php echo $heartbeat_settings['mode'] === 'default' ? '#2196f3' : '#dee2e6'; ?>; border-radius: 6px; margin-bottom: 10px; cursor: pointer;">
                        <input type="radio" name="heartbeat_mode" value="default" <?php checked($heartbeat_settings['mode'], 'default'); ?>>
                        <strong style="margin-left: 10px;">🔴 Default</strong> (WordPress štandard)
                        <div style="margin-left: 30px; font-size: 13px; color: #666; margin-top: 5px;">
                            Admin: 15s | Frontend: 15s | Editor: 15s<br>
                            <em>Vysoká server load - 240 req/hod per user v admin</em>
                        </div>
                    </label>
                    
                    <label style="display: block; padding: 15px; background: <?php echo $heartbeat_settings['mode'] === 'optimized' ? '#e3f2fd' : '#f8f9fa'; ?>; border: 2px solid <?php echo $heartbeat_settings['mode'] === 'optimized' ? '#2196f3' : '#dee2e6'; ?>; border-radius: 6px; margin-bottom: 10px; cursor: pointer;">
                        <input type="radio" name="heartbeat_mode" value="optimized" <?php checked($heartbeat_settings['mode'], 'optimized'); ?>>
                        <strong style="margin-left: 10px;">✅ Optimalizované</strong> (odporúčané)
                        <div style="margin-left: 30px; font-size: 13px; color: #666; margin-top: 5px;">
                            Admin: 30s | Frontend: VYPNUTÉ | Editor: 15s<br>
                            <em>Úspora 70% - ideálny pomer funkčnosť/výkon</em>
                        </div>
                    </label>
                    
                    <label style="display: block; padding: 15px; background: <?php echo $heartbeat_settings['mode'] === 'aggressive' ? '#e3f2fd' : '#f8f9fa'; ?>; border: 2px solid <?php echo $heartbeat_settings['mode'] === 'aggressive' ? '#2196f3' : '#dee2e6'; ?>; border-radius: 6px; margin-bottom: 10px; cursor: pointer;">
                        <input type="radio" name="heartbeat_mode" value="aggressive" <?php checked($heartbeat_settings['mode'], 'aggressive'); ?>>
                        <strong style="margin-left: 10px;">🚀 Aggressive</strong> (maximálna úspora)
                        <div style="margin-left: 30px; font-size: 13px; color: #666; margin-top: 5px;">
                            Admin: 60s | Frontend: VYPNUTÉ | Editor: 30s<br>
                            <em>Úspora 85% - pre weby s vysokou návštevnosťou</em>
                        </div>
                    </label>
                    
                    <label style="display: block; padding: 15px; background: <?php echo $heartbeat_settings['mode'] === 'disabled' ? '#e3f2fd' : '#f8f9fa'; ?>; border: 2px solid <?php echo $heartbeat_settings['mode'] === 'disabled' ? '#2196f3' : '#dee2e6'; ?>; border-radius: 6px; margin-bottom: 10px; cursor: pointer;">
                        <input type="radio" name="heartbeat_mode" value="disabled" <?php checked($heartbeat_settings['mode'], 'disabled'); ?>>
                        <strong style="margin-left: 10px;">⛔ Disabled</strong> (úplne vypnuté)
                        <div style="margin-left: 30px; font-size: 13px; color: #666; margin-top: 5px;">
                            Admin: VYPNUTÉ | Frontend: VYPNUTÉ | Editor: VYPNUTÉ<br>
                            <em style="color: #dc3545;">⚠️ Stratíte auto-save a post locking! Nie odporúčané.</em>
                        </div>
                    </label>
                </div>
                
                <div style="background: #fff3cd; padding: 15px; border-left: 4px solid #ffc107; border-radius: 4px; margin-bottom: 15px;">
                    <strong>💡 Čo sa stane?</strong>
                    <ul style="margin: 10px 0 0; padding-left: 20px; line-height: 1.8;">
                        <li><strong>Frontend vypnuté:</strong> Návštevníci negenerujú Heartbeat requesty (najväčšia úspora)</li>
                        <li><strong>Admin spomalený:</strong> Auto-save a notifikácie budú pomalšie (stále funkčné)</li>
                        <li><strong>Editor:</strong> Ponechaný rýchly pre smooth písanie článkov</li>
                        <li><strong>Post locking:</strong> Funguje aj pri 30-60s intervale</li>
                    </ul>
                </div>
                
                <button type="submit" name="wseo_save_heartbeat" class="button button-primary button-large">
                    💾 Uložiť nastavenia Heartbeat
                </button>
                
                <span style="margin-left: 15px; color: #28a745; font-weight: bold;" id="wseo-heartbeat-savings">
                    <?php if ($current_load['total_saved'] > 0): ?>
                        ✓ Aktuálna úspora: <?php echo $current_load['total_saved']; ?>% server load
                    <?php endif; ?>
                </span>
            </form>
        </div>
        
        <details style="background: rgba(255,255,255,0.3); padding: 15px; border-radius: 6px; margin-top: 15px;">
            <summary style="cursor: pointer; font-weight: bold; color: #333;">
                ℹ️ Čo je WordPress Heartbeat API?
            </summary>
            <div style="margin-top: 15px; line-height: 1.8; color: #333;">
                <p><strong>Heartbeat API</strong> je built-in WordPress systém pre real-time komunikáciu:</p>
                <ul style="padding-left: 20px;">
                    <li><strong>Auto-save:</strong> Automaticky ukladá rozpísané články každých 15s</li>
                    <li><strong>Post locking:</strong> Upozorní ak niekto iný edituje článok</li>
                    <li><strong>Session management:</strong> Kontroluje či ste prihlásený</li>
                    <li><strong>Notifications:</strong> Live notifikácie o komentároch, updates</li>
                </ul>
                <p style="margin-top: 10px;"><strong>Problém:</strong> Každý admin user generuje 240 requestov/hod. Pri 10 useroch = 2400 req/hod! 
                Na frontende s 100 návštevníkmi = 24000 req/hod! 🔥</p>
                <p><strong>Riešenie:</strong> Optimalizovaný režim vypne frontend a spomalí admin → úspora 70% bez straty funkčnosti.</p>
            </div>
        </details>
    </div>
    
    <!-- AJAX Call Log -->
    <div class="wseo-admin-card">
        <h2>📊 AJAX Log (posledných 24h)</h2>
        <p class="description">História background AJAX volaní na frontende.</p>
        
        <div style="max-height: 400px; overflow-y: auto; background: #f8f9fa; padding: 10px; border-radius: 4px;">
            <?php if (empty($ajax_log)): ?>
                <p style="text-align: center; color: #999; padding: 20px;">Žiadne AJAX volania zaznamenané</p>
            <?php else: ?>
                <table class="wp-list-table widefat" style="background: white;">
                    <thead>
                        <tr>
                            <th>Čas</th>
                            <th>Endpoint</th>
                            <th>Trvanie</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach (array_slice($ajax_log, 0, 50) as $log): ?>
                            <tr>
                                <td><?php echo date('H:i:s', $log['timestamp']); ?></td>
                                <td><code><?php echo esc_html($log['endpoint']); ?></code></td>
                                <td><?php echo number_format($log['duration'], 0); ?>ms</td>
                                <td>
                                    <?php if ($log['status'] === 'success'): ?>
                                        <span style="color: #28a745;">✓</span>
                                    <?php else: ?>
                                        <span style="color: #dc3545;">✗</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
        
        <p style="margin-top: 10px;">
            <button type="button" class="button" id="wseo-clear-ajax-log">
                Vymazať log
            </button>
        </p>
    </div>
    
    <div id="wseo-performance-result"></div>
</div>

<script>
var ajaxurl = '<?php echo admin_url('admin-ajax.php'); ?>';

jQuery(document).ready(function($) {
    // Start live monitoring
    $('#wseo-start-live-monitor').on('click', function() {
        var $btn = $(this);
        var $status = $('#wseo-live-status');
        var $message = $('#wseo-live-message');
        
        $btn.prop('disabled', true);
        $status.show();
        $message.text('Monitorujem pozadie... (zostáva 30s)');
        
        // Enable monitoring on frontend
        $.post(ajaxurl, {
            action: 'wseo_start_live_monitor',
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        });
        
        // Countdown
        var seconds = 30;
        var interval = setInterval(function() {
            seconds--;
            $message.text('Monitorujem pozadie... (zostáva ' + seconds + 's)');
            
            if (seconds <= 0) {
                clearInterval(interval);
                $message.text('Spracovávam dáta...');
                
                // Get results
                setTimeout(function() {
                    location.reload();
                }, 2000);
            }
        }, 1000);
    });
    
    // Refresh data
    $('#wseo-refresh-data').on('click', function() {
        location.reload();
    });
    
    // Disable process temporarily
    $('.wseo-disable-process').on('click', function() {
        var $btn = $(this);
        var hook = $btn.data('hook');
        
        if (!confirm('Vypnúť tento proces dočasne na 24 hodín?')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Vypínam...');
        
        $.post(ajaxurl, {
            action: 'wseo_disable_process',
            hook: hook,
            duration: 24,
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                $btn.closest('tr').fadeOut(300, function() {
                    location.reload();
                });
            } else {
                alert('Chyba: ' + response.data);
                $btn.prop('disabled', false).text('Vypnúť dočasne');
            }
        });
    });
    
    // Disable process permanently
    $('.wseo-disable-process-permanent').on('click', function() {
        var $btn = $(this);
        var hook = $btn.data('hook');
        
        if (!confirm('Vypnúť tento proces NATRVALO? Môžete ho kedykoľvek znova zapnúť.')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Vypínam...');
        
        $.post(ajaxurl, {
            action: 'wseo_disable_process',
            hook: hook,
            permanent: true,
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Chyba: ' + response.data);
                $btn.prop('disabled', false).text('Vypnúť natrvalo');
            }
        });
    });
    
    // Enable process
    $('.wseo-enable-process-permanent').on('click', function() {
        var $btn = $(this);
        var hook = $btn.data('hook');
        
        $btn.prop('disabled', true).text('Zapínam...');
        
        $.post(ajaxurl, {
            action: 'wseo_enable_process',
            hook: hook,
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Chyba: ' + response.data);
                $btn.prop('disabled', false).text('Zapnúť');
            }
        });
    });
    
    // Disable all safe processes
    $('#wseo-disable-all-safe').on('click', function() {
        if (!confirm('Vypnúť VŠETKY bezpečné procesy naraz? Odporúčame to iba ak ste si istí.')) {
            return;
        }
        
        var $btn = $(this);
        $btn.prop('disabled', true).text('Vypínam všetky...');
        
        $.post(ajaxurl, {
            action: 'wseo_disable_all_safe',
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                alert('Vypnutých: ' + response.data.disabled + ' procesov\nPotenciálna úspora: ' + response.data.saved + 's');
                location.reload();
            } else {
                alert('Chyba: ' + response.data);
                $btn.prop('disabled', false).text('Vypnúť všetky bezpečné procesy');
            }
        });
    });
    
    // Clear AJAX log
    $('#wseo-clear-ajax-log').on('click', function() {
        if (!confirm('Vymazať celý AJAX log?')) {
            return;
        }
        
        $.post(ajaxurl, {
            action: 'wseo_clear_ajax_log',
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                location.reload();
            }
        });
    });
    
    // Adjust frequency
    $('.wseo-adjust-frequency').on('change', function() {
        var $select = $(this);
        var hook = $select.data('hook');
        var delay = $select.val();
        
        if (!delay) {
            return;
        }
        
        var delayText = delay >= 1000 ? (delay / 1000) + 's' : delay + 'ms';
        
        if (!confirm('Zmeniť frekvenciu tohto procesu na ' + delayText + '?\n\nProces sa bude spúšťať menej často, čo zníži záťaž servera.')) {
            $select.val('');
            return;
        }
        
        $select.prop('disabled', true);
        
        $.post(ajaxurl, {
            action: 'wseo_adjust_frequency',
            hook: hook,
            delay: delay,
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                alert('✓ Frekvencia zmenená!\n\n' + response.data.message);
                location.reload();
            } else {
                alert('Chyba: ' + response.data);
                $select.prop('disabled', false).val('');
            }
        });
    });
    
    // Apply smart recommendation
    $('.wseo-apply-smart-recommendation').on('click', function() {
        var $btn = $(this);
        var hook = $btn.data('hook');
        var delay = $btn.data('delay');
        var action = $btn.data('action');
        var delayText = (delay / 1000) + 's';
        
        if (!confirm('🤖 Aplikovať AI odporúčanie?\n\nProces "' + action + '" bude throttlovaný na ' + delayText + '.\n\nToto zníži záťaž servera a zrýchli web.')) {
            return;
        }
        
        $btn.prop('disabled', true).text('Aplikujem...');
        
        $.post(ajaxurl, {
            action: 'wseo_adjust_frequency',
            hook: hook,
            delay: delay,
            nonce: '<?php echo wp_create_nonce('wseo_performance'); ?>'
        }, function(response) {
            if (response.success) {
                alert('✅ AI odporúčanie aplikované!\n\n' + response.data.message + '\n\n💡 Stránka sa reloadne aby sa zmeny prejavili.');
                location.reload();
            } else {
                alert('Chyba: ' + response.data);
                $btn.prop('disabled', false).text('⚡ Aplikovať odporúčanie');
            }
        });
    });
    
    // Scroll to section
    $('.wseo-scroll-to').on('click', function() {
        var target = $(this).data('target');
        var $target = $('#' + target);
        
        if ($target.length) {
            $('html, body').animate({
                scrollTop: $target.offset().top - 100
            }, 500);
            
            // Highlight effect
            $target.css('box-shadow', '0 0 20px rgba(255, 193, 7, 0.5)');
            setTimeout(function() {
                $target.css('box-shadow', '');
            }, 2000);
        }
    });
});
</script>

<style>
.wseo-admin-wrap {
    max-width: 1400px;
}

.wseo-admin-card {
    background: white;
    padding: 20px;
    margin-bottom: 20px;
    border-radius: 8px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.wseo-admin-card h2 {
    margin-top: 0;
    padding-bottom: 10px;
    border-bottom: 2px solid #f0f0f1;
}

.wseo-badge {
    display: inline-block;
    font-weight: 600;
    text-transform: uppercase;
}

.wp-list-table td {
    vertical-align: middle !important;
}
</style>
