// IndexNow by Webstudio.ltd - Full Version with i18n Support

// === TRANSLATIONS ===
const translations = {
  sk: {
    // Tabs
    tab_page: "Stránka",
    tab_sitemap: "Sitemap",
    tab_settings: "Nastavenia",
    
    // Page tab
    loading: "Načítavam...",
    page_not_indexable: "Táto stránka sa nedá indexovať",
    setup_title: "⚡ Prvé nastavenie (30 sekúnd)",
    setup_step1: "Stiahni súbor kliknutím nižšie",
    setup_step2: "Nahraj ho na svoj web (FTP/cPanel) do hlavného adresára",
    setup_step3: "Klikni \"Hotovo\"",
    btn_download: "📥 Stiahnuť verifikačný súbor",
    btn_done: "✓ Hotovo, súbor je na serveri",
    ready_text: "✓ Pripravené",
    change_link: "(zmeniť)",
    btn_index_page: "⚡ Indexovať túto stránku",
    
    // Sitemap tab
    sitemap_url_label: "Sitemap URL",
    sitemap_hint: "Alebo klikni na Auto-detect pre automatické vyhľadanie",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Načítať",
    detected_sitemaps: "Nájdené sitemapy:",
    stat_total: "Celkom",
    stat_sent: "Odoslané",
    stat_errors: "Chyby",
    filter_all: "Zobrazené: všetky URL",
    filter_success: "Zobrazené: úspešne odoslané",
    filter_error: "Zobrazené: URL s chybami",
    btn_index_all: "⚡ Indexovať všetky",
    btn_export: "📥 Export CSV",
    btn_stop: "⏹ Zastaviť",
    plugins_title: "Podporované SEO pluginy:",
    
    // Settings tab
    settings_key_title: "🔑 IndexNow kľúč",
    btn_regenerate: "🔄 Vygenerovať nový kľúč",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Voliteľné",
    gsc_key_label: "GSC API Kľúč / OAuth JSON",
    gsc_hint: "Pre automatické prepojenie s Google Search Console",
    bing_key_label: "Bing API Kľúč",
    bing_hint: "Nájdeš v Bing Webmaster → Settings → API Access",
    yandex_key_label: "Yandex API Kľúč",
    btn_save: "💾 Uložiť nastavenia",
    
    // Status messages
    status_sending: "Odosielam...",
    status_sent_to: "✅ Odoslané na",
    status_error_check_file: "❌ Chyba - skontroluj či súbor existuje na serveri",
    status_file_downloaded: "✓ Súbor stiahnutý!",
    status_setup_complete: "✓ Nastavenie dokončené!",
    status_searching: "Hľadám sitemapy...",
    status_no_sitemaps: "⚠️ Žiadne sitemapy nenájdené. Skús zadať URL manuálne.",
    status_found_sitemaps: "✓ Nájdených {count} sitemap(ov). Klikni na jednu pre načítanie.",
    status_loading_sitemap: "Načítavam sitemap...",
    status_loading_child: "Načítavam {count} child sitemap(ov)...",
    status_sitemap_empty: "Sitemap je prázdny alebo neplatný",
    status_loaded_urls: "✓ Načítaných {count} URL",
    status_complete_setup: "⚠️ Najprv dokonči nastavenie v záložke \"Stránka\"",
    status_complete: "✅ Dokončené! Odoslaných: {sent}, Chýb: {errors}",
    status_settings_saved: "✓ Nastavenia uložené",
    status_new_key: "✓ Nový kľúč vygenerovaný",
    status_no_urls_export: "⚠️ Žiadne URL na export",
    status_exported: "✓ Exportovaných {count} URL do CSV",
    
    // URL statuses
    url_pending: "čaká",
    url_success: "✓",
    url_error: "✗",
    no_urls: "Žiadne URL",
    no_success: "Zatiaľ žiadne úspešne odoslané URL",
    no_errors: "Žiadne chyby 🎉",
    and_more: "... a {count} ďalších",
    
    // Confirm
    confirm_new_key: "Naozaj vygenerovať nový kľúč? Budeš musieť nahrať nový súbor na server."
  },
  
  en: {
    tab_page: "Page",
    tab_sitemap: "Sitemap",
    tab_settings: "Settings",
    loading: "Loading...",
    page_not_indexable: "This page cannot be indexed",
    setup_title: "⚡ First setup (30 seconds)",
    setup_step1: "Download the file by clicking below",
    setup_step2: "Upload it to your website root folder (FTP/cPanel)",
    setup_step3: "Click \"Done\"",
    btn_download: "📥 Download verification file",
    btn_done: "✓ Done, file is on server",
    ready_text: "✓ Ready",
    change_link: "(change)",
    btn_index_page: "⚡ Index this page",
    sitemap_url_label: "Sitemap URL",
    sitemap_hint: "Or click Auto-detect to find automatically",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Load",
    detected_sitemaps: "Found sitemaps:",
    stat_total: "Total",
    stat_sent: "Sent",
    stat_errors: "Errors",
    filter_all: "Showing: all URLs",
    filter_success: "Showing: successfully sent",
    filter_error: "Showing: URLs with errors",
    btn_index_all: "⚡ Index all",
    btn_export: "📥 Export CSV",
    btn_stop: "⏹ Stop",
    plugins_title: "Supported SEO plugins:",
    settings_key_title: "🔑 IndexNow key",
    btn_regenerate: "🔄 Generate new key",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Optional",
    gsc_key_label: "GSC API Key / OAuth JSON",
    gsc_hint: "For automatic Google Search Console integration",
    bing_key_label: "Bing API Key",
    bing_hint: "Find in Bing Webmaster → Settings → API Access",
    yandex_key_label: "Yandex API Key",
    btn_save: "💾 Save settings",
    status_sending: "Sending...",
    status_sent_to: "✅ Sent to",
    status_error_check_file: "❌ Error - check if the file exists on server",
    status_file_downloaded: "✓ File downloaded!",
    status_setup_complete: "✓ Setup complete!",
    status_searching: "Searching for sitemaps...",
    status_no_sitemaps: "⚠️ No sitemaps found. Try entering URL manually.",
    status_found_sitemaps: "✓ Found {count} sitemap(s). Click one to load.",
    status_loading_sitemap: "Loading sitemap...",
    status_loading_child: "Loading {count} child sitemap(s)...",
    status_sitemap_empty: "Sitemap is empty or invalid",
    status_loaded_urls: "✓ Loaded {count} URLs",
    status_complete_setup: "⚠️ Complete setup in \"Page\" tab first",
    status_complete: "✅ Complete! Sent: {sent}, Errors: {errors}",
    status_settings_saved: "✓ Settings saved",
    status_new_key: "✓ New key generated",
    status_no_urls_export: "⚠️ No URLs to export",
    status_exported: "✓ Exported {count} URLs to CSV",
    url_pending: "pending",
    url_success: "✓",
    url_error: "✗",
    no_urls: "No URLs",
    no_success: "No successfully sent URLs yet",
    no_errors: "No errors 🎉",
    and_more: "... and {count} more",
    confirm_new_key: "Generate new key? You will need to upload a new file to server."
  },
  
  de: {
    tab_page: "Seite",
    tab_sitemap: "Sitemap",
    tab_settings: "Einstellungen",
    loading: "Laden...",
    page_not_indexable: "Diese Seite kann nicht indexiert werden",
    setup_title: "⚡ Ersteinrichtung (30 Sekunden)",
    setup_step1: "Datei herunterladen",
    setup_step2: "Auf Ihren Server hochladen (FTP/cPanel) ins Hauptverzeichnis",
    setup_step3: "Klicken Sie \"Fertig\"",
    btn_download: "📥 Verifizierungsdatei herunterladen",
    btn_done: "✓ Fertig, Datei ist auf dem Server",
    ready_text: "✓ Bereit",
    change_link: "(ändern)",
    btn_index_page: "⚡ Diese Seite indexieren",
    sitemap_url_label: "Sitemap URL",
    sitemap_hint: "Oder Auto-detect klicken zum automatischen Suchen",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Laden",
    detected_sitemaps: "Gefundene Sitemaps:",
    stat_total: "Gesamt",
    stat_sent: "Gesendet",
    stat_errors: "Fehler",
    filter_all: "Angezeigt: alle URLs",
    filter_success: "Angezeigt: erfolgreich gesendet",
    filter_error: "Angezeigt: URLs mit Fehlern",
    btn_index_all: "⚡ Alle indexieren",
    btn_export: "📥 CSV Export",
    btn_stop: "⏹ Stopp",
    plugins_title: "Unterstützte SEO-Plugins:",
    settings_key_title: "🔑 IndexNow Schlüssel",
    btn_regenerate: "🔄 Neuen Schlüssel generieren",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Optional",
    gsc_key_label: "GSC API Schlüssel / OAuth JSON",
    gsc_hint: "Für automatische Google Search Console Integration",
    bing_key_label: "Bing API Schlüssel",
    bing_hint: "Finden Sie unter Bing Webmaster → Settings → API Access",
    yandex_key_label: "Yandex API Schlüssel",
    btn_save: "💾 Einstellungen speichern",
    status_sending: "Senden...",
    status_sent_to: "✅ Gesendet an",
    status_error_check_file: "❌ Fehler - prüfen Sie ob die Datei auf dem Server existiert",
    status_file_downloaded: "✓ Datei heruntergeladen!",
    status_setup_complete: "✓ Einrichtung abgeschlossen!",
    status_searching: "Suche nach Sitemaps...",
    status_no_sitemaps: "⚠️ Keine Sitemaps gefunden. Versuchen Sie die URL manuell einzugeben.",
    status_found_sitemaps: "✓ {count} Sitemap(s) gefunden. Klicken Sie eine zum Laden.",
    status_loading_sitemap: "Lade Sitemap...",
    status_loading_child: "Lade {count} Child-Sitemap(s)...",
    status_sitemap_empty: "Sitemap ist leer oder ungültig",
    status_loaded_urls: "✓ {count} URLs geladen",
    status_complete_setup: "⚠️ Zuerst Einrichtung im \"Seite\" Tab abschließen",
    status_complete: "✅ Fertig! Gesendet: {sent}, Fehler: {errors}",
    status_settings_saved: "✓ Einstellungen gespeichert",
    status_new_key: "✓ Neuer Schlüssel generiert",
    status_no_urls_export: "⚠️ Keine URLs zum Exportieren",
    status_exported: "✓ {count} URLs als CSV exportiert",
    url_pending: "wartend",
    url_success: "✓",
    url_error: "✗",
    no_urls: "Keine URLs",
    no_success: "Noch keine erfolgreich gesendeten URLs",
    no_errors: "Keine Fehler 🎉",
    and_more: "... und {count} weitere",
    confirm_new_key: "Neuen Schlüssel generieren? Sie müssen eine neue Datei auf den Server hochladen."
  },
  
  fr: {
    tab_page: "Page",
    tab_sitemap: "Sitemap",
    tab_settings: "Paramètres",
    loading: "Chargement...",
    page_not_indexable: "Cette page ne peut pas être indexée",
    setup_title: "⚡ Première configuration (30 secondes)",
    setup_step1: "Téléchargez le fichier ci-dessous",
    setup_step2: "Envoyez-le sur votre serveur (FTP/cPanel) dans le dossier racine",
    setup_step3: "Cliquez \"Terminé\"",
    btn_download: "📥 Télécharger le fichier de vérification",
    btn_done: "✓ Terminé, fichier sur le serveur",
    ready_text: "✓ Prêt",
    change_link: "(modifier)",
    btn_index_page: "⚡ Indexer cette page",
    sitemap_url_label: "URL du Sitemap",
    sitemap_hint: "Ou cliquez sur Auto-detect pour rechercher automatiquement",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Charger",
    detected_sitemaps: "Sitemaps trouvés:",
    stat_total: "Total",
    stat_sent: "Envoyés",
    stat_errors: "Erreurs",
    filter_all: "Affichés: tous les URLs",
    filter_success: "Affichés: envoyés avec succès",
    filter_error: "Affichés: URLs avec erreurs",
    btn_index_all: "⚡ Indexer tout",
    btn_export: "📥 Export CSV",
    btn_stop: "⏹ Arrêter",
    plugins_title: "Plugins SEO supportés:",
    settings_key_title: "🔑 Clé IndexNow",
    btn_regenerate: "🔄 Générer nouvelle clé",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Optionnel",
    gsc_key_label: "Clé API GSC / OAuth JSON",
    gsc_hint: "Pour l'intégration automatique avec Google Search Console",
    bing_key_label: "Clé API Bing",
    bing_hint: "Trouvez dans Bing Webmaster → Settings → API Access",
    yandex_key_label: "Clé API Yandex",
    btn_save: "💾 Enregistrer",
    status_sending: "Envoi...",
    status_sent_to: "✅ Envoyé à",
    status_error_check_file: "❌ Erreur - vérifiez si le fichier existe sur le serveur",
    status_file_downloaded: "✓ Fichier téléchargé!",
    status_setup_complete: "✓ Configuration terminée!",
    status_searching: "Recherche de sitemaps...",
    status_no_sitemaps: "⚠️ Aucun sitemap trouvé. Essayez d'entrer l'URL manuellement.",
    status_found_sitemaps: "✓ {count} sitemap(s) trouvé(s). Cliquez pour charger.",
    status_loading_sitemap: "Chargement du sitemap...",
    status_loading_child: "Chargement de {count} sitemap(s) enfant(s)...",
    status_sitemap_empty: "Sitemap vide ou invalide",
    status_loaded_urls: "✓ {count} URLs chargées",
    status_complete_setup: "⚠️ Terminez d'abord la configuration dans l'onglet \"Page\"",
    status_complete: "✅ Terminé! Envoyés: {sent}, Erreurs: {errors}",
    status_settings_saved: "✓ Paramètres enregistrés",
    status_new_key: "✓ Nouvelle clé générée",
    status_no_urls_export: "⚠️ Aucune URL à exporter",
    status_exported: "✓ {count} URLs exportées en CSV",
    url_pending: "en attente",
    url_success: "✓",
    url_error: "✗",
    no_urls: "Aucune URL",
    no_success: "Aucune URL envoyée avec succès pour l'instant",
    no_errors: "Aucune erreur 🎉",
    and_more: "... et {count} de plus",
    confirm_new_key: "Générer une nouvelle clé? Vous devrez télécharger un nouveau fichier sur le serveur."
  },
  
  it: {
    tab_page: "Pagina",
    tab_sitemap: "Sitemap",
    tab_settings: "Impostazioni",
    loading: "Caricamento...",
    page_not_indexable: "Questa pagina non può essere indicizzata",
    setup_title: "⚡ Prima configurazione (30 secondi)",
    setup_step1: "Scarica il file cliccando sotto",
    setup_step2: "Caricalo sul tuo server (FTP/cPanel) nella cartella principale",
    setup_step3: "Clicca \"Fatto\"",
    btn_download: "📥 Scarica file di verifica",
    btn_done: "✓ Fatto, file sul server",
    ready_text: "✓ Pronto",
    change_link: "(modifica)",
    btn_index_page: "⚡ Indicizza questa pagina",
    sitemap_url_label: "URL Sitemap",
    sitemap_hint: "Oppure clicca Auto-detect per cercare automaticamente",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Carica",
    detected_sitemaps: "Sitemap trovate:",
    stat_total: "Totale",
    stat_sent: "Inviati",
    stat_errors: "Errori",
    filter_all: "Mostrati: tutti gli URL",
    filter_success: "Mostrati: inviati con successo",
    filter_error: "Mostrati: URL con errori",
    btn_index_all: "⚡ Indicizza tutti",
    btn_export: "📥 Esporta CSV",
    btn_stop: "⏹ Ferma",
    plugins_title: "Plugin SEO supportati:",
    settings_key_title: "🔑 Chiave IndexNow",
    btn_regenerate: "🔄 Genera nuova chiave",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Opzionale",
    gsc_key_label: "Chiave API GSC / OAuth JSON",
    gsc_hint: "Per l'integrazione automatica con Google Search Console",
    bing_key_label: "Chiave API Bing",
    bing_hint: "Trova in Bing Webmaster → Settings → API Access",
    yandex_key_label: "Chiave API Yandex",
    btn_save: "💾 Salva impostazioni",
    status_sending: "Invio...",
    status_sent_to: "✅ Inviato a",
    status_error_check_file: "❌ Errore - verifica se il file esiste sul server",
    status_file_downloaded: "✓ File scaricato!",
    status_setup_complete: "✓ Configurazione completata!",
    status_searching: "Ricerca sitemap...",
    status_no_sitemaps: "⚠️ Nessuna sitemap trovata. Prova a inserire l'URL manualmente.",
    status_found_sitemaps: "✓ Trovate {count} sitemap. Clicca per caricare.",
    status_loading_sitemap: "Caricamento sitemap...",
    status_loading_child: "Caricamento {count} sitemap figlie...",
    status_sitemap_empty: "Sitemap vuota o non valida",
    status_loaded_urls: "✓ Caricati {count} URL",
    status_complete_setup: "⚠️ Prima completa la configurazione nella scheda \"Pagina\"",
    status_complete: "✅ Completato! Inviati: {sent}, Errori: {errors}",
    status_settings_saved: "✓ Impostazioni salvate",
    status_new_key: "✓ Nuova chiave generata",
    status_no_urls_export: "⚠️ Nessun URL da esportare",
    status_exported: "✓ {count} URL esportati in CSV",
    url_pending: "in attesa",
    url_success: "✓",
    url_error: "✗",
    no_urls: "Nessun URL",
    no_success: "Nessun URL inviato con successo ancora",
    no_errors: "Nessun errore 🎉",
    and_more: "... e altri {count}",
    confirm_new_key: "Generare nuova chiave? Dovrai caricare un nuovo file sul server."
  },
  
  ru: {
    tab_page: "Страница",
    tab_sitemap: "Sitemap",
    tab_settings: "Настройки",
    loading: "Загрузка...",
    page_not_indexable: "Эту страницу нельзя индексировать",
    setup_title: "⚡ Первая настройка (30 секунд)",
    setup_step1: "Скачайте файл, нажав ниже",
    setup_step2: "Загрузите его на сервер (FTP/cPanel) в корневую папку",
    setup_step3: "Нажмите \"Готово\"",
    btn_download: "📥 Скачать файл верификации",
    btn_done: "✓ Готово, файл на сервере",
    ready_text: "✓ Готово",
    change_link: "(изменить)",
    btn_index_page: "⚡ Индексировать страницу",
    sitemap_url_label: "URL Sitemap",
    sitemap_hint: "Или нажмите Auto-detect для автоматического поиска",
    btn_autodetect: "🔍 Auto-detect",
    btn_load: "📥 Загрузить",
    detected_sitemaps: "Найденные sitemap:",
    stat_total: "Всего",
    stat_sent: "Отправлено",
    stat_errors: "Ошибки",
    filter_all: "Показаны: все URL",
    filter_success: "Показаны: успешно отправленные",
    filter_error: "Показаны: URL с ошибками",
    btn_index_all: "⚡ Индексировать все",
    btn_export: "📥 Экспорт CSV",
    btn_stop: "⏹ Остановить",
    plugins_title: "Поддерживаемые SEO плагины:",
    settings_key_title: "🔑 Ключ IndexNow",
    btn_regenerate: "🔄 Сгенерировать новый ключ",
    settings_gsc: "🔍 Google Search Console",
    settings_bing: "🅱️ Bing Webmaster Tools",
    settings_yandex: "🔵 Yandex Webmaster",
    optional: "Опционально",
    gsc_key_label: "GSC API Ключ / OAuth JSON",
    gsc_hint: "Для автоматической интеграции с Google Search Console",
    bing_key_label: "Bing API Ключ",
    bing_hint: "Найдите в Bing Webmaster → Settings → API Access",
    yandex_key_label: "Yandex API Ключ",
    btn_save: "💾 Сохранить настройки",
    status_sending: "Отправка...",
    status_sent_to: "✅ Отправлено на",
    status_error_check_file: "❌ Ошибка - проверьте существует ли файл на сервере",
    status_file_downloaded: "✓ Файл скачан!",
    status_setup_complete: "✓ Настройка завершена!",
    status_searching: "Поиск sitemap...",
    status_no_sitemaps: "⚠️ Sitemap не найдены. Попробуйте ввести URL вручную.",
    status_found_sitemaps: "✓ Найдено {count} sitemap. Нажмите для загрузки.",
    status_loading_sitemap: "Загрузка sitemap...",
    status_loading_child: "Загрузка {count} дочерних sitemap...",
    status_sitemap_empty: "Sitemap пустой или недействительный",
    status_loaded_urls: "✓ Загружено {count} URL",
    status_complete_setup: "⚠️ Сначала завершите настройку во вкладке \"Страница\"",
    status_complete: "✅ Завершено! Отправлено: {sent}, Ошибок: {errors}",
    status_settings_saved: "✓ Настройки сохранены",
    status_new_key: "✓ Новый ключ сгенерирован",
    status_no_urls_export: "⚠️ Нет URL для экспорта",
    status_exported: "✓ Экспортировано {count} URL в CSV",
    url_pending: "ожидание",
    url_success: "✓",
    url_error: "✗",
    no_urls: "Нет URL",
    no_success: "Пока нет успешно отправленных URL",
    no_errors: "Нет ошибок 🎉",
    and_more: "... и ещё {count}",
    confirm_new_key: "Сгенерировать новый ключ? Вам нужно будет загрузить новый файл на сервер."
  }
};

// Current language
let currentLang = 'sk';

// Get translation with optional replacements
function t(key, replacements = {}) {
  let text = translations[currentLang]?.[key] || translations['en']?.[key] || key;
  
  for (const [k, v] of Object.entries(replacements)) {
    text = text.replace(`{${k}}`, v);
  }
  
  return text;
}

// Apply translations to all elements with data-i18n
function applyTranslations() {
  document.querySelectorAll('[data-i18n]').forEach(el => {
    const key = el.getAttribute('data-i18n');
    el.textContent = t(key);
  });
  
  // Update active language flag
  document.querySelectorAll('.lang-flag').forEach(flag => {
    flag.classList.toggle('active', flag.dataset.lang === currentLang);
  });
}

// === MAIN APP ===
document.addEventListener('DOMContentLoaded', async () => {
  // Elements
  const tabs = document.querySelectorAll('.tab');
  const tabContents = document.querySelectorAll('.tab-content');
  const langFlags = document.querySelectorAll('.lang-flag');
  
  const urlBox = document.getElementById('currentUrl');
  const setupMode = document.getElementById('setupMode');
  const readyMode = document.getElementById('readyMode');
  const keyDisplay = document.getElementById('keyDisplay');
  const downloadBtn = document.getElementById('downloadBtn');
  const doneBtn = document.getElementById('doneBtn');
  const submitBtn = document.getElementById('submitBtn');
  const resetBtn = document.getElementById('resetBtn');
  const pageStatus = document.getElementById('pageStatus');
  
  const sitemapUrl = document.getElementById('sitemapUrl');
  const autoDetectBtn = document.getElementById('autoDetectBtn');
  const loadSitemapBtn = document.getElementById('loadSitemapBtn');
  const detectedSitemaps = document.getElementById('detectedSitemaps');
  const sitemapList = document.getElementById('sitemapList');
  const sitemapResults = document.getElementById('sitemapResults');
  const urlList = document.getElementById('urlList');
  const submitAllBtn = document.getElementById('submitAllBtn');
  const stopBtn = document.getElementById('stopBtn');
  const exportCsvBtn = document.getElementById('exportCsvBtn');
  const sitemapStatus = document.getElementById('sitemapStatus');
  const totalUrlsEl = document.getElementById('totalUrls');
  const sentUrlsEl = document.getElementById('sentUrls');
  const errorUrlsEl = document.getElementById('errorUrls');
  const progressFill = document.getElementById('progressFill');
  const filterInfo = document.getElementById('filterInfo');
  const statBoxes = document.querySelectorAll('.stat');
  
  const settingsKeyDisplay = document.getElementById('settingsKeyDisplay');
  const regenerateKeyBtn = document.getElementById('regenerateKeyBtn');
  const gscKey = document.getElementById('gscKey');
  const bingKey = document.getElementById('bingKey');
  const yandexKey = document.getElementById('yandexKey');
  const saveSettingsBtn = document.getElementById('saveSettingsBtn');
  const settingsStatus = document.getElementById('settingsStatus');
  
  // State
  let currentUrl = null;
  let currentDomain = null;
  let apiKey = null;
  let urlData = [];
  let currentFilter = 'all';
  let isRunning = false;
  
  // Initialize
  await init();
  
  async function init() {
    // Load saved language or detect from browser
    const data = await chrome.storage.local.get(['language', 'indexnow_key', 'indexnow_setup', 'gsc_key', 'bing_key', 'yandex_key']);
    
    if (data.language) {
      currentLang = data.language;
    } else {
      // Auto-detect from browser
      const browserLang = navigator.language.slice(0, 2);
      if (translations[browserLang]) {
        currentLang = browserLang;
      }
    }
    
    applyTranslations();
    
    // Load current URL
    try {
      const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (tab && tab.url && tab.url.startsWith('http')) {
        currentUrl = tab.url;
        currentDomain = new URL(currentUrl).origin;
        urlBox.textContent = currentUrl;
        sitemapUrl.value = currentDomain + '/sitemap.xml';
      } else {
        urlBox.textContent = t('page_not_indexable');
        submitBtn.disabled = true;
        doneBtn.disabled = true;
      }
    } catch (e) {
      urlBox.textContent = t('page_not_indexable');
    }
    
    // API Key
    if (data.indexnow_key) {
      apiKey = data.indexnow_key;
    } else {
      apiKey = generateKey();
      await chrome.storage.local.set({ indexnow_key: apiKey });
    }
    
    keyDisplay.textContent = apiKey;
    settingsKeyDisplay.textContent = apiKey;
    
    if (data.indexnow_setup) {
      setupMode.classList.add('hidden');
      readyMode.classList.remove('hidden');
    }
    
    if (data.gsc_key) gscKey.value = data.gsc_key;
    if (data.bing_key) bingKey.value = data.bing_key;
    if (data.yandex_key) yandexKey.value = data.yandex_key;
    
    setupEventListeners();
  }
  
  function setupEventListeners() {
    // Tab navigation
    tabs.forEach(tab => {
      tab.addEventListener('click', () => {
        tabs.forEach(t => t.classList.remove('active'));
        tabContents.forEach(c => c.classList.remove('active'));
        tab.classList.add('active');
        document.getElementById('tab-' + tab.dataset.tab).classList.add('active');
      });
    });
    
    // Language switcher
    langFlags.forEach(flag => {
      flag.addEventListener('click', async () => {
        currentLang = flag.dataset.lang;
        await chrome.storage.local.set({ language: currentLang });
        applyTranslations();
        if (currentFilter) setFilter(currentFilter);
      });
    });
    
    // Page tab
    downloadBtn.addEventListener('click', downloadKeyFile);
    doneBtn.addEventListener('click', completeSetup);
    resetBtn.addEventListener('click', resetSetup);
    submitBtn.addEventListener('click', submitCurrentPage);
    
    // Sitemap tab
    autoDetectBtn.addEventListener('click', autoDetectSitemaps);
    loadSitemapBtn.addEventListener('click', () => loadSitemap(sitemapUrl.value.trim()));
    submitAllBtn.addEventListener('click', submitAllUrls);
    stopBtn.addEventListener('click', () => { isRunning = false; });
    exportCsvBtn.addEventListener('click', exportToCsv);
    
    statBoxes.forEach(box => {
      box.addEventListener('click', () => setFilter(box.dataset.filter));
    });
    
    // Settings tab
    regenerateKeyBtn.addEventListener('click', regenerateKey);
    saveSettingsBtn.addEventListener('click', saveSettings);
  }
  
  // === LANGUAGE-AWARE STATUS ===
  function showStatus(el, type, key, replacements = {}) {
    el.className = 'status ' + type;
    el.textContent = t(key, replacements);
  }
  
  function showStatusText(el, type, text) {
    el.className = 'status ' + type;
    el.textContent = text;
  }
  
  // === FILTER ===
  function setFilter(filter) {
    currentFilter = filter;
    statBoxes.forEach(box => box.classList.toggle('active', box.dataset.filter === filter));
    filterInfo.textContent = t('filter_' + filter);
    renderUrlList();
  }
  
  function getFilteredUrls() {
    if (currentFilter === 'all') return urlData;
    return urlData.filter(item => item.status === currentFilter);
  }
  
  function renderUrlList() {
    const filtered = getFilteredUrls();
    
    if (filtered.length === 0) {
      const key = currentFilter === 'all' ? 'no_urls' : currentFilter === 'success' ? 'no_success' : 'no_errors';
      urlList.innerHTML = `<div class="url-item"><span class="url-text" style="color:#64748b;">${t(key)}</span></div>`;
      return;
    }
    
    const toShow = filtered.slice(0, 100);
    urlList.innerHTML = toShow.map(item => `
      <div class="url-item">
        <span class="url-text" title="${item.url}">${truncateUrl(item.url, 40)}</span>
        <span class="badge ${item.status}">${t('url_' + item.status)}</span>
      </div>
    `).join('');
    
    if (filtered.length > 100) {
      urlList.innerHTML += `<div class="url-item"><span class="url-text" style="color:#64748b;">${t('and_more', { count: filtered.length - 100 })}</span></div>`;
    }
  }
  
  function updateStats() {
    totalUrlsEl.textContent = urlData.length;
    sentUrlsEl.textContent = urlData.filter(u => u.status === 'success').length;
    errorUrlsEl.textContent = urlData.filter(u => u.status === 'error').length;
    const processed = urlData.filter(u => u.status !== 'pending').length;
    progressFill.style.width = urlData.length > 0 ? ((processed / urlData.length) * 100) + '%' : '0%';
  }
  
  // === PAGE TAB ===
  function downloadKeyFile() {
    const blob = new Blob([apiKey], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = apiKey + '.txt';
    a.click();
    URL.revokeObjectURL(url);
    showStatus(pageStatus, 'success', 'status_file_downloaded');
  }
  
  async function completeSetup() {
    await chrome.storage.local.set({ indexnow_setup: true });
    setupMode.classList.add('hidden');
    readyMode.classList.remove('hidden');
    showStatus(pageStatus, 'success', 'status_setup_complete');
  }
  
  async function resetSetup() {
    await chrome.storage.local.remove(['indexnow_setup']);
    setupMode.classList.remove('hidden');
    readyMode.classList.add('hidden');
    pageStatus.className = 'status';
  }
  
  async function submitCurrentPage() {
    if (!currentUrl) return;
    submitBtn.disabled = true;
    showStatus(pageStatus, 'loading', 'status_sending');
    
    const result = await submitToIndexNow([currentUrl]);
    
    if (result.success > 0) {
      showStatusText(pageStatus, 'success', `${t('status_sent_to')} ${result.endpoints.join(', ')}`);
    } else {
      showStatus(pageStatus, 'error', 'status_error_check_file');
    }
    submitBtn.disabled = false;
  }
  
  // === SITEMAP TAB ===
  async function autoDetectSitemaps() {
    if (!currentDomain) return;
    
    autoDetectBtn.disabled = true;
    showStatus(sitemapStatus, 'loading', 'status_searching');
    
    let detectedPlugin = null;
    let detectedSitemap = null;
    
    // 1. Fetch homepage to detect active SEO plugin
    try {
      const response = await fetch(currentUrl || currentDomain);
      const html = await response.text();
      
      // Detect by HTML comments and meta tags
      if (html.includes('Yoast SEO') || html.includes('yoast-schema')) {
        detectedPlugin = 'Yoast SEO';
        detectedSitemap = '/sitemap_index.xml';
      } else if (html.includes('Rank Math') || html.includes('rank-math')) {
        detectedPlugin = 'RankMath';
        detectedSitemap = '/sitemap_index.xml';
      } else if (html.includes('SEOPress') || html.includes('seopress')) {
        detectedPlugin = 'SEOPress';
        detectedSitemap = '/sitemaps.xml';
      } else if (html.includes('All in One SEO') || html.includes('aioseo')) {
        detectedPlugin = 'All in One SEO';
        detectedSitemap = '/sitemap.xml';
      } else if (html.includes('The SEO Framework')) {
        detectedPlugin = 'SEO Framework';
        detectedSitemap = '/sitemap.xml';
      } else if (html.includes('Squirrly')) {
        detectedPlugin = 'Squirrly SEO';
        detectedSitemap = '/sitemap.xml';
      } else if (html.includes('SmartCrawl')) {
        detectedPlugin = 'SmartCrawl';
        detectedSitemap = '/sitemap.xml';
      } else if (html.includes('WP Meta SEO') || html.includes('flavor="flavor"')) {
        detectedPlugin = 'WP Meta SEO';
        detectedSitemap = '/sitemap.xml';
      }
    } catch (e) {}
    
    // 2. If plugin detected, verify sitemap exists
    if (detectedPlugin && detectedSitemap) {
      try {
        const response = await fetch(currentDomain + detectedSitemap, { method: 'HEAD' });
        if (response.ok) {
          autoDetectBtn.disabled = false;
          sitemapUrl.value = currentDomain + detectedSitemap;
          
          detectedSitemaps.classList.remove('hidden');
          sitemapList.innerHTML = `
            <div class="sitemap-item" data-url="${currentDomain + detectedSitemap}">
              <span class="sitemap-name">${detectedSitemap}</span>
              <span class="sitemap-type">${detectedPlugin}</span>
            </div>
          `;
          
          sitemapList.querySelector('.sitemap-item').addEventListener('click', () => {
            loadSitemap(currentDomain + detectedSitemap);
          });
          
          showStatus(sitemapStatus, 'success', 'status_found_sitemaps', { count: 1 });
          return;
        }
      } catch (e) {}
    }
    
    // 3. Fallback: try common sitemaps in priority order
    const fallbackOrder = [
      { url: '/sitemap_index.xml', name: 'Yoast/RankMath' },
      { url: '/sitemaps.xml', name: 'SEOPress' },
      { url: '/sitemap.xml', name: 'Standard' },
      { url: '/wp-sitemap.xml', name: 'WordPress Core' }
    ];
    
    for (const sm of fallbackOrder) {
      try {
        const response = await fetch(currentDomain + sm.url, { method: 'HEAD' });
        if (response.ok) {
          autoDetectBtn.disabled = false;
          sitemapUrl.value = currentDomain + sm.url;
          
          detectedSitemaps.classList.remove('hidden');
          sitemapList.innerHTML = `
            <div class="sitemap-item" data-url="${currentDomain + sm.url}">
              <span class="sitemap-name">${sm.url}</span>
              <span class="sitemap-type">${sm.name}</span>
            </div>
          `;
          
          sitemapList.querySelector('.sitemap-item').addEventListener('click', () => {
            loadSitemap(currentDomain + sm.url);
          });
          
          showStatus(sitemapStatus, 'success', 'status_found_sitemaps', { count: 1 });
          return;
        }
      } catch (e) {}
    }
    
    // 4. Check robots.txt as last resort
    try {
      const robotsSitemaps = await extractSitemapsFromRobots(currentDomain + '/robots.txt');
      if (robotsSitemaps.length > 0) {
        const sm = robotsSitemaps[0];
        autoDetectBtn.disabled = false;
        sitemapUrl.value = sm.url;
        
        detectedSitemaps.classList.remove('hidden');
        sitemapList.innerHTML = `
          <div class="sitemap-item" data-url="${sm.url}">
            <span class="sitemap-name">${sm.url.replace(currentDomain, '')}</span>
            <span class="sitemap-type">robots.txt</span>
          </div>
        `;
        
        sitemapList.querySelector('.sitemap-item').addEventListener('click', () => {
          loadSitemap(sm.url);
        });
        
        showStatus(sitemapStatus, 'success', 'status_found_sitemaps', { count: 1 });
        return;
      }
    } catch (e) {}
    
    autoDetectBtn.disabled = false;
    showStatus(sitemapStatus, 'warning', 'status_no_sitemaps');
  }
  
  async function extractSitemapsFromRobots(robotsUrl) {
    const sitemaps = [];
    try {
      const response = await fetch(robotsUrl);
      const text = await response.text();
      const lines = text.split('\n');
      for (const line of lines) {
        const match = line.match(/^sitemap:\s*(.+)/i);
        if (match) {
          sitemaps.push({ url: match[1].trim(), name: 'robots.txt' });
        }
      }
    } catch (e) {}
    return sitemaps;
  }
  
  async function loadSitemap(url) {
    if (!url) return;
    
    loadSitemapBtn.disabled = true;
    autoDetectBtn.disabled = true;
    showStatus(sitemapStatus, 'loading', 'status_loading_sitemap');
    
    try {
      const urls = await fetchSitemap(url, 0);
      
      if (urls.length === 0) {
        showStatus(sitemapStatus, 'error', 'status_sitemap_empty');
        loadSitemapBtn.disabled = false;
        autoDetectBtn.disabled = false;
        return;
      }
      
      urlData = [...new Set(urls)].map(u => ({ url: u, status: 'pending' }));
      currentFilter = 'all';
      statBoxes.forEach(box => box.classList.toggle('active', box.dataset.filter === 'all'));
      filterInfo.textContent = t('filter_all');
      
      sitemapResults.classList.remove('hidden');
      updateStats();
      renderUrlList();
      
      showStatus(sitemapStatus, 'success', 'status_loaded_urls', { count: urlData.length });
    } catch (e) {
      showStatusText(sitemapStatus, 'error', '❌ ' + e.message);
    }
    
    loadSitemapBtn.disabled = false;
    autoDetectBtn.disabled = false;
  }
  
  async function fetchSitemap(url, depth = 0) {
    if (depth > 5) return [];
    
    const response = await fetch(url);
    if (!response.ok) throw new Error('Cannot load sitemap');
    
    const text = await response.text();
    const parser = new DOMParser();
    const xml = parser.parseFromString(text, 'text/xml');
    
    if (xml.querySelector('parsererror')) throw new Error('Invalid XML');
    
    const sitemapLocs = xml.querySelectorAll('sitemap > loc');
    if (sitemapLocs.length > 0) {
      showStatus(sitemapStatus, 'loading', 'status_loading_child', { count: sitemapLocs.length });
      let allUrls = [];
      for (const loc of sitemapLocs) {
        try {
          const childUrls = await fetchSitemap(loc.textContent.trim(), depth + 1);
          allUrls = allUrls.concat(childUrls);
        } catch (e) {}
      }
      return allUrls;
    }
    
    return Array.from(xml.querySelectorAll('url > loc')).map(loc => loc.textContent.trim());
  }
  
  async function submitAllUrls() {
    if (urlData.length === 0) return;
    
    const data = await chrome.storage.local.get(['indexnow_setup']);
    if (!data.indexnow_setup) {
      showStatus(sitemapStatus, 'warning', 'status_complete_setup');
      return;
    }
    
    isRunning = true;
    submitAllBtn.classList.add('hidden');
    stopBtn.classList.remove('hidden');
    loadSitemapBtn.disabled = true;
    autoDetectBtn.disabled = true;
    
    showStatus(sitemapStatus, 'loading', 'status_sending');
    urlData.forEach(item => item.status = 'pending');
    updateStats();
    renderUrlList();
    
    const byDomain = {};
    for (const item of urlData) {
      try {
        const host = new URL(item.url).hostname;
        if (!byDomain[host]) byDomain[host] = [];
        byDomain[host].push(item);
      } catch (e) {}
    }
    
    for (const [host, items] of Object.entries(byDomain)) {
      if (!isRunning) break;
      
      for (let i = 0; i < items.length; i += 100) {
        if (!isRunning) break;
        
        const batch = items.slice(i, i + 100);
        const result = await submitToIndexNow(batch.map(item => item.url));
        const status = result.success > 0 ? 'success' : 'error';
        batch.forEach(item => item.status = status);
        
        updateStats();
        renderUrlList();
        await sleep(500);
      }
    }
    
    isRunning = false;
    submitAllBtn.classList.remove('hidden');
    stopBtn.classList.add('hidden');
    loadSitemapBtn.disabled = false;
    autoDetectBtn.disabled = false;
    
    const sent = urlData.filter(u => u.status === 'success').length;
    const errors = urlData.filter(u => u.status === 'error').length;
    showStatus(sitemapStatus, 'success', 'status_complete', { sent, errors });
  }
  
  function exportToCsv() {
    const filtered = getFilteredUrls();
    
    if (filtered.length === 0) {
      showStatus(sitemapStatus, 'warning', 'status_no_urls_export');
      return;
    }
    
    const headers = ['URL', 'Status'];
    const rows = filtered.map(item => {
      const statusText = item.status === 'success' ? t('stat_sent') : item.status === 'error' ? t('stat_errors') : t('url_pending');
      return `"${item.url}","${statusText}"`;
    });
    
    const csvContent = [headers.join(','), ...rows].join('\n');
    const blob = new Blob(['\ufeff' + csvContent], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    
    const filterNames = { all: 'all-urls', success: 'sent-urls', error: 'error-urls' };
    a.href = url;
    a.download = `indexnow-${filterNames[currentFilter]}-${new Date().toISOString().slice(0, 10)}.csv`;
    a.click();
    URL.revokeObjectURL(url);
    
    showStatus(sitemapStatus, 'success', 'status_exported', { count: filtered.length });
  }
  
  // === SETTINGS TAB ===
  async function regenerateKey() {
    if (!confirm(t('confirm_new_key'))) return;
    
    apiKey = generateKey();
    await chrome.storage.local.set({ indexnow_key: apiKey, indexnow_setup: false });
    
    keyDisplay.textContent = apiKey;
    settingsKeyDisplay.textContent = apiKey;
    setupMode.classList.remove('hidden');
    readyMode.classList.add('hidden');
    
    showStatus(settingsStatus, 'success', 'status_new_key');
  }
  
  async function saveSettings() {
    await chrome.storage.local.set({
      gsc_key: gscKey.value.trim(),
      bing_key: bingKey.value.trim(),
      yandex_key: yandexKey.value.trim()
    });
    showStatus(settingsStatus, 'success', 'status_settings_saved');
  }
  
  // === INDEXNOW API ===
  async function submitToIndexNow(urlList) {
    const host = new URL(urlList[0]).hostname;
    const endpoints = [
      { name: 'IndexNow', url: 'https://api.indexnow.org/indexnow' },
      { name: 'Bing', url: 'https://www.bing.com/indexnow' },
      { name: 'Yandex', url: 'https://yandex.com/indexnow' }
    ];
    
    const results = { success: 0, endpoints: [] };
    
    for (const ep of endpoints) {
      try {
        const response = await fetch(ep.url, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json; charset=utf-8' },
          body: JSON.stringify({
            host,
            key: apiKey,
            keyLocation: `https://${host}/${apiKey}.txt`,
            urlList
          })
        });
        
        if (response.ok || response.status === 200 || response.status === 202) {
          results.success++;
          results.endpoints.push(ep.name);
        }
      } catch (e) {}
    }
    
    return results;
  }
  
  // === UTILITIES ===
  function generateKey() {
    const chars = '0123456789abcdef';
    const array = new Uint8Array(32);
    crypto.getRandomValues(array);
    return Array.from(array).map(b => chars[b % 16]).join('');
  }
  
  function truncateUrl(url, max) {
    return url.length > max ? url.substring(0, max) + '...' : url;
  }
  
  function sleep(ms) {
    return new Promise(r => setTimeout(r, ms));
  }
});
